using System;
using System.ComponentModel;
using vu.ch.argee.BaseClasses;
using vu.ch.argee.WindowsAPI.WrapperClasses;

namespace vu.ch.argee.MemoryMapping
{
	/// <summary>
	/// Maps a part of a file (view) into the address space of the calling process.
	/// Use UnmanagedMemoryStream and BinaryReader/BinaryWriter 
	/// resp. any other StreamReader/StreamWriter
	/// classes to access the mapped memory.
	/// </summary>
	public class FileMappingView: BaseDisposableObject, IUnmanagedMemoryResource, IBufferedResource 
	{
		#region ------------ fields ------------

		private IntPtr memoryPointer;
		private IntPtr baseAddress;
		private uint numberOfBytes; 
		private uint adjustedNumberOfBytesMapped;
		private MappingAccess viewAccess;

		#endregion

		#region ------------ contructors ------------

		/// <summary>
		/// Maps the whole file in memory. 
		/// MappingAccess is same as in the underlying FileMapping.
		/// </summary>
		/// <param name="fileMapping">The FileMapping object to map in physical memory.</param>
		/// <remarks>
		/// <b>NOTE: This constructor only works for files smaller than 2GB. For bigger files use the overloaded method where you specify fileOffset and numberOfBytesToMap.</b><BR/>
		/// The size of a file view is limited to the largest available contiguous block of unreserved virtual memory. This is at most 2 GB minus the virtual memory already reserved by the process.
		/// </remarks>
		/// <exception cref="InvalidOperationException">The <i>fileMapping</i> cannot be completely mapped or has an unknown size.</exception>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		public FileMappingView(FileMapping fileMapping)
		{
			if ((fileMapping.MaxiumumSize == 0) || (fileMapping.MaxiumumSize > int.MaxValue))
			{
				throw new InvalidOperationException("The fileMapping cannot be completely mapped or has an unknown size. Use the overloaded FileMappingView constructor and specify the size to map.");
			}
			Initialize(fileMapping.Handle, fileMapping.Access, 0, (uint)fileMapping.MaxiumumSize);
		}

		/// <summary>
		/// Maps the whole file in memory.
		/// </summary>
		/// <param name="fileMapping">The FileMapping to map in physical memory.</param>
		/// <param name="viewAccess">The access protection for the view.</param>
		/// <exception cref="InvalidOperationException">The <i>fileMapping</i> cannot be completely mapped or has an unknown size.</exception>
		/// <remarks>
		/// <b>NOTE: Windows 95/98/ME the MappingAccess must have same value as in the underlying FileMapping.</b><br/>
		/// <b>NOTE: This constructor only works for files smaller than 2GB. For bigger files use the overloaded method where you specify fileOffset and numberOfBytesToMap.</b>
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		public FileMappingView(FileMapping fileMapping, MappingAccess viewAccess)
		{
			if ((fileMapping.MaxiumumSize == 0) || (fileMapping.MaxiumumSize > int.MaxValue))
			{
				throw new InvalidOperationException("The fileMapping cannot be completely mapped or has an unknown size. Use the overloaded FileMappingView constructor and specify the size to map.");
			}
			Initialize(fileMapping.Handle, viewAccess, 0, (uint)fileMapping.MaxiumumSize);
		}

		/// <summary>
		/// Maps a part of the file in memory. Useful for big files. 
		/// MappingAccess is same as in the underlying FileMapping.
		/// </summary>
		/// <param name="fileMapping">The FileMapping to map in physical memory.</param>
		/// <param name="fileOffset">Offset of the first byte to map.</param>
		/// <param name="numberOfBytesToMap">Number of bytes to map in this view.</param>
		/// <remarks>
		/// NOTE: You don't have care about the allocation granularity. This is done internally.<br/> 
		/// The size of a file view is limited to the largest available contiguous block of unreserved virtual memory. This is at most 2 GB minus the virtual memory already reserved by the process.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		public FileMappingView(FileMapping fileMapping, ulong fileOffset, uint numberOfBytesToMap)
		{
			Initialize(fileMapping.Handle, fileMapping.Access, fileOffset, numberOfBytesToMap);
		}

		/// <summary>
		/// Maps a part of the file in memory. Useful for big files. 
		/// </summary>
		/// <param name="fileMapping">The FileMapping to map in physical memory.</param>
		/// <param name="viewAccess">The access protection for the view.</param>
		/// <param name="fileOffset">Offset of the first byte to map.</param>
		/// <param name="numberOfBytesToMap">Number of bytes to map in this view.</param>
		/// <remarks>
		/// <b>NOTE: Windows 95/98/ME the MappingAccess must have same value as in the underlying FileMapping.</b><br/>
		/// NOTE: You don't have care about the allocation granularity. This is done internally.<br/>
		/// The size of a file view is limited to the largest available contiguous block of unreserved virtual memory. This is at most 2 GB minus the virtual memory already reserved by the process.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		public FileMappingView(FileMapping fileMapping, MappingAccess viewAccess, ulong fileOffset, uint numberOfBytesToMap)
		{
			if ((numberOfBytesToMap == 0) || (numberOfBytesToMap > int.MaxValue))
			{
				throw new ArgumentOutOfRangeException("numberOfBytesToMap", numberOfBytesToMap, "Value must be greater than zero and less than 2GB.");
			}
			Initialize(fileMapping.Handle, viewAccess, fileOffset, numberOfBytesToMap);
		}

		/// <summary>
		/// Maps a part of the file in memory. 
		/// This constructor uses a system handle to specify the file mapping object that could be passed from a different process by the DuplicateHandle function.
		/// This constructor is also internally used to create a view of the mapped file in physical memory.
		/// </summary>
		/// <param name="fileMappingObjectHandle">The handle of the system file mapping object to map in physical memory.</param>
		/// <param name="viewAccess">The access protection for the view.</param>
		/// <param name="fileOffset">Offset of the first byte to map.</param>
		/// <param name="numberOfBytesToMap">Number of bytes to map in this view.</param>
		/// <remarks>
		/// <b>NOTE: Windows 95/98/ME the MappingAccess must have same value as in the underlying FileMapping.</b><br/>
		/// NOTE: You don't have care about the allocation granularity. This is done internally.<br/>
		/// The size of a file view is limited to the largest available contiguous block of unreserved virtual memory. This is at most 2 GB minus the virtual memory already reserved by the process.
		/// </remarks>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		/// <include file='examples.xml' path='MemoryMapping/FileMappingView[@name="example1"]/*' />
		public FileMappingView(IntPtr fileMappingObjectHandle, MappingAccess viewAccess, ulong fileOffset, uint numberOfBytesToMap)
		{
			if ((numberOfBytesToMap == 0) || (numberOfBytesToMap > int.MaxValue))
			{
				throw new ArgumentOutOfRangeException("numberOfBytesToMap", numberOfBytesToMap, "Value must be greater than zero and less than 2GB.");
			}
			Initialize(fileMappingObjectHandle, viewAccess, fileOffset, numberOfBytesToMap);
		}

		#endregion

		#region ------------ private methods ------------

		private void Initialize(IntPtr fileMappingObjectHandle, MappingAccess viewAccess, ulong fileOffset, uint numberOfBytesToMap)
		{
			uint baseAddressOffset;

			this.numberOfBytes = numberOfBytesToMap;
			this.viewAccess = viewAccess;
			
			baseAddressOffset = ManagedFileMappingAdaptor.GetAllocationGranularityOffset(ref fileOffset);
			adjustedNumberOfBytesMapped = numberOfBytesToMap + baseAddressOffset;
			memoryPointer = ManagedFileMappingAdaptor.MapViewOfFile(fileMappingObjectHandle, MappingAccessConverter.ToFileMapAccess(viewAccess), fileOffset, adjustedNumberOfBytesMapped);
			baseAddress = new IntPtr((int)memoryPointer + baseAddressOffset);
		}

		#endregion

		#region ------------ protected methods ------------

		/// <summary>
		/// Unmaps the FileMapping (ViewOfFile).
		/// </summary>
		protected override void DisposeUnmanagedResources()
		{
			// the ViewOfFile doesn't have a handle - it's referenced by the base address
			ManagedFileMappingAdaptor.UnmapViewOfFile(memoryPointer);
		}

		/// <summary>
		/// Disposes managed resources.
		/// </summary>
		/// <remarks>The current implementaion of this method does nothing.</remarks>
		protected override void DisposeManagedResources()
		{
		}

		#endregion

		#region ------------ public methods ------------

		/// <summary>
		/// Flushes the changes to disk. If the file is not local 
		/// it's still not sure that it is completely written to disk.
		/// </summary>
		/// <exception cref="Win32Exception">A Win32API function failed.</exception>
		public void Flush()
		{
			if (!IsDisposed) // allow this call for cascaded Streams and StreamReaders/StreamWriters
			{
				ManagedFileMappingAdaptor.FlushViewOfFile(memoryPointer, adjustedNumberOfBytesMapped);
			}
		}

		/// <summary>
		/// Closes the FileMappingView and any system resources associated with the FileMappingView. 
		/// </summary>
		public void Close()
		{
			Dispose();
		}

		#endregion

		#region ------------ properties ------------

		/// <summary>
		/// MappingAccess value used to create this object.
		/// </summary>
		/// <value>The MappingAccess value used to create this object.</value>
		public MappingAccess Access 
		{
			get
			{
				CheckDisposed();
				return viewAccess;
			}
		}

		/// <summary>
		/// Base address of the file mapping view. 
		/// Can be used to access the memory directly by the methods of the Marshal class.
		/// </summary>
		/// <value>The Base address of the file mapping view.</value>
		/// <remarks> 
		/// This value may differ from the base address of the MapViewOfFile system object which is adjusted to the allocation granularity.
		/// </remarks>
		public IntPtr BaseAddress
		{
			get
			{
				CheckDisposed();
				return baseAddress;
			}
		}

		/// <summary>
		/// Number of bytes mapped into physical memory.
		/// </summary>
		/// <value>The Number of bytes mapped into physical memory.</value>
		public uint Length 
		{
			get
			{
				CheckDisposed();
				return numberOfBytes;
			}
		}

		/// <summary>
		/// Returns true if the view is writeable. 
		/// If the data is written back into the file depends on the MappingAccess 
		/// of the FileMappingView and the underlying FileMapping.
		/// </summary>
		/// <value><b>true</b> if the view is writeable.</value>
		public bool CanWrite
		{
			get
			{ 
				CheckDisposed();
				return viewAccess != MappingAccess.ReadOnly;
			}
		}

		#endregion
	}
}
