using System;
using System.IO;
using System.Runtime.InteropServices;
using System.ComponentModel; // contains definition of Win32Exception

namespace vu.ch.argee.WindowsAPI.WrapperClasses
{
	/// <summary>
	/// Wrappes the file mapping functions of the StorageAPI
	/// with exception handlers and type converters.
	/// </summary>
	public class ManagedFileMappingAdaptor
	{
		private static uint allocationGranularity;

		// Initializes the static variables
		static ManagedFileMappingAdaptor()
		{
			SYSTEM_INFO systemInformation;

			systemInformation = new SYSTEM_INFO();
			SystemInformationAPI.GetSystemInfo(ref systemInformation);
			allocationGranularity = systemInformation.dwAllocationGranularity;
		}

		/// <summary>
		/// Constructor is private.
		/// </summary>
		private ManagedFileMappingAdaptor()
		{}

		private static IntPtr CheckResult(IntPtr result)
		{
			if (result == IntPtr.Zero)
			{
				if (Marshal.GetLastWin32Error() != StorageAPI.ERROR_ALREADY_EXISTS)
				{
					throw new Win32Exception();
				}
			}
			return result;
		}

		private static bool CheckResult(bool result)
		{
			if (!result)
			{
				if (Marshal.GetLastWin32Error() != StorageAPI.ERROR_ALREADY_EXISTS)
				{
					throw new Win32Exception();
				}
			}
			return result;
		}

		/// <summary>
		/// Converts the FileMode value to the FILE_MODE value used by the Storage API.
		/// </summary>
		/// <param name="fileMode">the FileMode value to convert</param>
		/// <returns>the corresponding FILE_MODE value</returns>
		internal static FILE_MODE ConvertFileMode(FileMode fileMode)
		{
			FILE_MODE result;

			switch(fileMode)
			{
				case FileMode.Append: result = FILE_MODE.OPEN_EXISTING;
					break;
				case FileMode.Create: result = FILE_MODE.CREATE_ALWAYS;
					break;
				case FileMode.CreateNew: result = FILE_MODE.CREATE_NEW;
					break;
				case FileMode.Open: result = FILE_MODE.OPEN_EXISTING;
					break;
				case FileMode.OpenOrCreate: result = FILE_MODE.OPEN_ALWAYS;
					break;
				case FileMode.Truncate: result = FILE_MODE.TRUNCATE_EXISTING;
					break;
				default:
					throw new InvalidOperationException("Invalid FileMode");
			}

			return result;
		}

		/// <summary>
		/// Converts the FileAccess value to the FILE_ACCESS value used by the Storage API.
		/// </summary>
		/// <param name="fileAccess">the FileAccess value to convert</param>
		/// <returns>the corresponding FILE_ACCESS value</returns>
		internal static FILE_ACCESS ConvertFileAccess(FileAccess fileAccess)
		{
			FILE_ACCESS result;

			switch(fileAccess)
			{
				case FileAccess.Read: result = FILE_ACCESS.GENERIC_READ;
					break;
				case FileAccess.ReadWrite: result = FILE_ACCESS.GENERIC_READ | FILE_ACCESS.GENERIC_WRITE;
					break;
				case FileAccess.Write: result = FILE_ACCESS.GENERIC_WRITE;
					break;
				default:
					throw new InvalidOperationException("Invalid FileAccess");
			}

			return result;
		}

		/// <summary>
		/// Converts the FileShare value to the FILE_SHARE value used by the Storage API.
		/// </summary>
		/// <param name="fileShare">the FileShare value to convert</param>
		/// <returns>the corresponding FILE_SHARE value</returns>
		internal static FILE_SHARE ConvertFileShare(FileShare fileShare)
		{
			FILE_SHARE result;

			switch(fileShare)
			{
				case FileShare.None: result = 0;
					break;
				case FileShare.Read: result = FILE_SHARE.READ;
					break;
				case FileShare.ReadWrite: result = FILE_SHARE.READ | FILE_SHARE.WRITE;
					break;
				case FileShare.Write: result = FILE_SHARE.WRITE;
					break;
				default:
					throw new InvalidOperationException("Invalid FileAccess");
			}

			return result;
		}

		/// <summary>
		/// Creates or opens a file, directory, physical disk, volume, console buffer, tape drive, communications resource, mailslot, or named pipe.
		/// </summary>
		/// <param name="fileName">The name of the file to create or open.</param>
		/// <param name="mode">A FileMode constant that determines how to open or create the file.</param>
		/// <param name="access">A FileAccess constant that determines how the file can be accessed.</param>
		/// <param name="share">A FileShare constant that determines how the file will be shared by processes.</param>
		/// <returns>A handle that can be used to access the system object.</returns>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr CreateFile(string fileName, FileMode mode, FileAccess access, FileShare share)
		{
			IntPtr result;

			fileName = Path.GetFullPath(fileName);

			// Attention: CreateFile returns INVALID_HANDLE when it fails
			result = StorageAPI.CreateFile(fileName, ConvertFileAccess(access), ConvertFileShare(share), null, ConvertFileMode(mode), FILE_ATTRIBUTE.NORMAL, IntPtr.Zero);
			if (result == StorageAPI.INVALID_HANDLE_VALUE)
			{
				throw new Win32Exception();
			}

			return result;
		}

		/// <summary>
		/// Creates or opens a named or unnamed file mapping object for the specified file.
		/// </summary>
		/// <param name="hFile">Handle to the file from which to create a mapping object.</param>
		/// <param name="securityAttributes">A SECURITY_ATTRIBUTES object that determines whether the returned handle can be inherited by child processes. If <I>securityAttributes</I> is <b>null</b>, the handle cannot be inherited.</param>
		/// <param name="protection">Protection desired for the file view, when the file is mapped.</param>
		/// <param name="maximumSize">Maximum size of the file mapping object.</param>
		/// <param name="name">The name of the mapping object. If this parameter matches the name of an existing named mapping object, the function requests access to the mapping object with the protection specified by <I>protection</I>.</param>
		/// <returns>Handle to the file mapping object. If the object existed before the function call, the function returns a handle to the existing object (with its current size, not the specified size).</returns>
		/// <remarks>
		/// An attempt to map a file with a length of zero in this manner fails with an error code of ERROR_FILE_INVALID.
		/// </remarks>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr CreateFileMapping(IntPtr hFile, SECURITY_ATTRIBUTES securityAttributes, PAGE_PROTECTION protection, ulong maximumSize, string name)
		{
			return CheckResult(StorageAPI.CreateFileMapping(hFile, securityAttributes, protection, (uint)(maximumSize >> 32), (uint)maximumSize, name));
		}

		/// <summary>
		/// Creates or opens a named or unnamed file mapping object for the specified file.
		/// </summary>
		/// <param name="hFile">Handle to the file from which to create a mapping object.</param>
		/// <param name="protection">Protection desired for the file view, when the file is mapped.</param>
		/// <param name="maximumSize">Maximum size of the file mapping object.</param>
		/// <param name="name">The name of the mapping object. If this parameter matches the name of an existing named mapping object, the function requests access to the mapping object with the protection specified by <I>protection</I>.</param>
		/// <returns>Handle to the file mapping object. If the object existed before the function call, the function returns a handle to the existing object (with its current size, not the specified size).</returns>
		/// <remarks>
		/// NOTE: The handle returned by this method cannot be inhertied.
		/// An attempt to map a file with a length of zero in this manner fails with an error code of ERROR_FILE_INVALID.
		/// </remarks>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr CreateFileMapping(IntPtr hFile, PAGE_PROTECTION protection, ulong maximumSize, string name)
		{
			return CheckResult(StorageAPI.CreateFileMapping(hFile, null, protection, (uint)(maximumSize >> 32), (uint)maximumSize, name));
		}

		/// <summary>
		/// Creates or opens a named or unnamed file mapping object backed up by the paging file.
		/// </summary>
		/// <param name="protection">Protection desired for the file view, when the file is mapped.</param>
		/// <param name="maximumSize">Maximum size of the file mapping object.</param>
		/// <param name="name">The name of the mapping object. If this parameter matches the name of an existing named mapping object, the function requests access to the mapping object with the protection specified by <I>protection</I>.</param>
		/// <returns>Handle to the file mapping object. If the object existed before the function call, the function returns a handle to the existing object (with its current size, not the specified size).</returns>
		/// <remarks>
		/// An attempt to map a file with a length of zero in this manner fails with an error code of ERROR_FILE_INVALID.
		/// </remarks>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr CreatePagingFileMapping(PAGE_PROTECTION protection, ulong maximumSize, string name)
		{
			return CheckResult(StorageAPI.CreateFileMapping(StorageAPI.INVALID_HANDLE_VALUE, null, protection, (uint)(maximumSize >> 32), (uint)maximumSize, name));
		}

		/// <summary>
		/// Opens a named file mapping object.
		/// </summary>
		/// <param name="desiredAccess">Access to the file mapping object. This <I>desiredAccess</I> is checked against any security descriptor on the target file mapping object.</param>
		/// <param name="inheritHandle">If this parameter is <b>true</b>, the new process inherits the handle. Otherwise, it does not.</param>
		/// <param name="name">Name of the file mapping object to be opened.</param>
		/// <returns>An open handle to the specified file mapping object.</returns>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr OpenFileMapping(FILE_MAP_ACCESS desiredAccess, bool inheritHandle, string name)
		{
			return CheckResult(StorageAPI.OpenFileMapping((uint)desiredAccess, inheritHandle, name)); 
		}

		/// <summary>
		/// Maps a view of a file into the address space of the calling process.
		/// </summary>
		/// <param name="fileMappingObject">Handle to an open file mapping object.</param>
		/// <param name="desiredAccess">The type of access to the file view.</param>
		/// <param name="fileOffset">The file offset where mapping is to begin. The <I>fileOffset</I> must specify an offset within the file that matches the system's memory allocation granularity, or the function fails</param>
		/// <param name="numberOfBytesToMap">Number of bytes of the file to map. If this parameter is zero, the entire file is mapped.</param>
		/// <returns>A pointer to the starting address of the mapped view.</returns>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr MapViewOfFile(IntPtr fileMappingObject, FILE_MAP_ACCESS desiredAccess, ulong fileOffset, uint numberOfBytesToMap)
		{
			return CheckResult(StorageAPI.MapViewOfFile(fileMappingObject, (uint)desiredAccess, (uint)(fileOffset >> 32), (uint)fileOffset, numberOfBytesToMap));
		}

		/// <summary>
		/// Writes a byte range within a mapped view of a file to the disk.
		/// </summary>
		/// <param name="baseAddress">The base address of the byte range to be flushed to the disk representation of the mapped file.</param>
		/// <param name="numberOfBytesToFlush">Number of bytes to flush.</param>
		/// <remarks>
		/// If <I>numberOfBytesToFlush</I> is zero, the file is flushed from the base address to the end of the mapping.
		/// </remarks>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static void FlushViewOfFile(IntPtr baseAddress, uint numberOfBytesToFlush)
		{
			CheckResult(StorageAPI.FlushViewOfFile(baseAddress, numberOfBytesToFlush));
		}

		/// <summary>
		/// Unmaps a mapped view of a file from the calling process's address space.
		/// </summary>
		/// <param name="baseAddress">the base address of the mapped view of a file that is to be unmapped.</param>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static void UnmapViewOfFile(IntPtr baseAddress)
		{
			CheckResult(StorageAPI.UnmapViewOfFile(baseAddress));
		}

		/// <summary>
		/// Closes the handle of the specified object.
		/// </summary>
		/// <param name="hObject">the handle to close</param>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static void CloseHandle(IntPtr hObject)
		{
			CheckResult(StorageAPI.CloseHandle(hObject));
		}

		/// <summary>
		/// Duplicates an object handle from or to a foreign process.
		/// </summary>
		/// <param name="sourceProcessHandle">the Handle to the process with the handle to duplicate.</param>
		/// <param name="sourceHandle">the handle to duplicate.</param>
		/// <param name="targetProcessHandle">the Handle to the process that is to receive the duplicated handle. The handle must have the PROCESS_DUP_HANDLE access right.</param>
		/// <param name="inheritHandle">indicates whether the handle is inheritable.</param>
		/// <returns>The duplicated handle. This handle value is valid in the context of the target process.</returns>
		/// <exception cref="Win32Exception">A system exception occured.</exception>
		public static IntPtr DuplicateHandle(IntPtr sourceProcessHandle, IntPtr sourceHandle, IntPtr targetProcessHandle, bool inheritHandle)
		{
			IntPtr targetHandle;

			CheckResult(StorageAPI.DuplicateHandle(sourceProcessHandle, sourceHandle, targetProcessHandle, out targetHandle, 0, inheritHandle, DUPLICATE_OPTIONS.DUPLICATE_SAME_ACCESS));

			return targetHandle;
		}

		/*
		/// <summary>
		/// Adjusts a value to the allocation granulatiry of the operating system.
		/// </summary>
		/// <param name="value">The value to adjust to the allocation granularity.</param>
		/// <returns>The adjusted value.</returns>
		public static uint RoundUpToAllocationGranularity(uint value)
		{
			uint mod;
			uint result;

			mod = value % allocationGranularity;
			if (mod == 0)
			{
				result = value;
			}
			else
			{
				result = value - mod + allocationGranularity;
			}

			return result;
		}
		*/

		/// <summary>
		/// Adjusts the value to to the allocation granulatiry of the operating system and returns the offset.
		/// </summary>
		/// <param name="value">The value to adjust to the allocation granularity.</param>
		/// <returns>The offset of the value to the allocation granulatiry.</returns>
		public static uint GetAllocationGranularityOffset(ref ulong value)
		{
			uint result;

			result = (uint)(value % (ulong)allocationGranularity);
			value = value - result;

			return result;
		}
	}
}
