using System;
using System.IO;
using vu.ch.argee.WindowsAPI;
using vu.ch.argee.WindowsAPI.WrapperClasses;

namespace vu.ch.argee.MemoryMapping
{
	/// <summary>
	/// Defines constants for readonly, read/write, or read/writecopy access to a memory mapped file
	/// </summary>
	public enum MappingAccess
	{
		/// <summary>
		/// Read-only access.
		/// </summary>
		ReadOnly,
		/// <summary>
		/// Read and write access.
		/// </summary>
		ReadWrite,
		/// <summary>
		/// Copy-on-write access.
		/// </summary>
		WriteCopy
	}

	/// <summary>
	/// Helper class to convert the MappingAccess enum in the StorageAPI specific values.
	/// </summary>
	public class MappingAccessConverter
	{
		/// <summary>
		/// Looks up the corresponding FileAccess for a MappingAccess value.
		/// </summary>
		/// <param name="mappingAccess">The MappingAccess value to convert.</param>
		/// <returns>The corresponding FileAccess value.</returns>
		public static FileAccess ToFileAccess(MappingAccess mappingAccess)
		{
			FileAccess fileAccess;

			switch (mappingAccess)
			{
				case MappingAccess.ReadOnly: fileAccess = FileAccess.Read;
					break;
				case MappingAccess.ReadWrite: fileAccess = FileAccess.ReadWrite;
					break;
				case MappingAccess.WriteCopy: fileAccess = FileAccess.ReadWrite;
					break;
				default:
					throw new InvalidOperationException("Invalid MappingAccess");
			}

			return fileAccess;
		}

		/// <summary>
		/// Looks up the corresponding PAGE_PROTECTION for a MappingAccess value.
		/// </summary>
		/// <param name="mappingAccess">The MappingAccess value to convert.</param>
		/// <returns>The corresponding PAGE_PROTECTION value.</returns>
		public static PAGE_PROTECTION ToPageProtection(MappingAccess mappingAccess)
		{
			PAGE_PROTECTION protection;

			switch (mappingAccess)
			{
				case MappingAccess.ReadOnly: protection = PAGE_PROTECTION.PAGE_READONLY;
					break;
				case MappingAccess.ReadWrite: protection = PAGE_PROTECTION.PAGE_READWRITE;
					break;
				case MappingAccess.WriteCopy: protection = PAGE_PROTECTION.PAGE_WRITECOPY;
					break;
				default:
					throw new InvalidOperationException("Invalid MappingAccess");
			}
			return protection;
		}

		/// <summary>
		/// Looks up the corresponding FileMapAccess for a MappingAccess value.
		/// </summary>
		/// <param name="mappingAccess">The MappingAccess value to convert.</param>
		/// <returns>The corresponding FileMapAccess value.</returns>
		public static  FILE_MAP_ACCESS ToFileMapAccess(MappingAccess mappingAccess)
		{
			FILE_MAP_ACCESS fileMapAccess;

			switch (mappingAccess)
			{
				case MappingAccess.ReadOnly: fileMapAccess = FILE_MAP_ACCESS.FILE_MAP_READ;
					break;
				case MappingAccess.ReadWrite: fileMapAccess = FILE_MAP_ACCESS.FILE_MAP_READ | FILE_MAP_ACCESS.FILE_MAP_WRITE;
					break;
				case MappingAccess.WriteCopy: fileMapAccess = FILE_MAP_ACCESS.FILE_MAP_COPY;
					break;
				default:
					throw new InvalidOperationException("Invalid MappingAccess");
			}
			return fileMapAccess;
		}
	}
}
