using System;
using System.Runtime.InteropServices;
using vu.ch.argee.WindowsAPI;

namespace vu.ch.argee.MemoryMapping
{
	/// <summary>
	/// Helper class to access unmanaged memory
	/// </summary>
	public class UnmanagedMemoryAccess
	{
		#region ------------ fields ------------

		private IUnmanagedMemory unmanagedMemory;
		private UnmanagedByteAccessor byteAccessor;

		#endregion

		#region ------------ constructors ------------

		/// <summary>
		/// Provides low level memory operations for unmanaged memory.
		/// </summary>
		/// <param name="unmanagedMemory">A reference to an IUnmanagedMemory interface.</param>
		/// <example>
		/// <code>
		/// using(fileMapping = FileMapping.CreateReadWrite(@"..\..\TestData.txt", 0))
		///	{
		///		using(view = new FileMappingView(fileMapping, MappingAccess.ReadWrite))
		///		{
		///			UnmanagedMemoryAccess memoryAccess;
		///			
		///			memory = myFileMappingView;
		///			memoryAccess = new UnmanagedMemoryAccess(view);
		///			memoryAccess.FillMemory(3,7, (byte)'A');
		///			memoryAccess.MoveMemory(22, 30, 100);		
		///		}
		///	}
		///	</code>
		/// </example>
		public UnmanagedMemoryAccess(IUnmanagedMemory unmanagedMemory)
		{
			this.unmanagedMemory = unmanagedMemory;		
			byteAccessor = new UnmanagedByteAccessor(unmanagedMemory);
		}

		#endregion

		#region ------------ public methods ------------

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="address"></param>
		/// <returns></returns>
		public IntPtr CheckAddress(IntPtr address)
		{
			int baseAddr;

			baseAddr = (int)unmanagedMemory.BaseAddress;
			if ((int)address >= (baseAddr + unmanagedMemory.Length))
			{
				throw new IndexOutOfRangeException();		
			}

			return address;
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="address"></param>
		/// <param name="length"></param>
		/// <returns></returns>
		public IntPtr CheckAddress(IntPtr address, uint length)
		{
			int baseAddr;

			baseAddr = (int)unmanagedMemory.BaseAddress;
			if ((int)address >= (baseAddr + unmanagedMemory.Length))
			{
				throw new IndexOutOfRangeException();		
			}
			if (((int)address + length) >= (baseAddr + unmanagedMemory.Length))
			{
				throw new IndexOutOfRangeException();		
			}

			return address;
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="offset"></param>
		/// <returns></returns>
		public IntPtr GetAddressFromOffset(uint offset)
		{
			uint baseAddr;
			uint addr;

			baseAddr = (uint)unmanagedMemory.BaseAddress;
			addr = baseAddr + offset;
			if (offset >= unmanagedMemory.Length)
			{
				throw new IndexOutOfRangeException();		
			}

			return (IntPtr)addr;
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="offset"></param>
		/// <param name="length"></param>
		/// <returns></returns>
		public IntPtr GetAddressFromOffset(uint offset, uint length)
		{
			uint baseAddr;
			uint addr;

			baseAddr = (uint)unmanagedMemory.BaseAddress;
			addr = baseAddr + offset;
			/*
			Cannot happen:
			if (offset >= unmanagedMemory.Length)
			{
				throw new IndexOutOfRangeException();		
			}
			*/
			if ((offset + length) >= unmanagedMemory.Length)
			{
				throw new IndexOutOfRangeException();		
			}

			return (IntPtr)addr;
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="destinationOffset"></param>
		/// <param name="length"></param>
		/// <param name="value"></param>
		public void FillMemory(uint destinationOffset, uint length, byte value)
		{
			MemoryManagementAPI.FillMemory(GetAddressFromOffset(destinationOffset, length), length, value);
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="destinationOffset"></param>
		/// <param name="sourceOffset"></param>
		/// <param name="length"></param>
		public void MoveMemory(uint destinationOffset, uint sourceOffset, uint length)
		{
			uint baseAddr;
			uint maxTargetOffset;

			baseAddr = (uint)unmanagedMemory.BaseAddress;
			maxTargetOffset = (uint)(unmanagedMemory.Length - length);
			if ((destinationOffset >= maxTargetOffset) || (sourceOffset >= maxTargetOffset))
			{
				throw new IndexOutOfRangeException();
			}
			MemoryManagementAPI.MoveMemory(baseAddr + destinationOffset, baseAddr + sourceOffset, length);
		}

		/// <summary>
		/// To be done.
		/// </summary>
		/// <param name="destinationOffset"></param>
		/// <param name="length"></param>
		public void ZeroMemory(uint destinationOffset, uint length)
		{
			MemoryManagementAPI.SecureZeroMemory(GetAddressFromOffset(destinationOffset, length), length);
		}

		#endregion

		#region ------------ properties ------------

		/// <summary>
		/// Gets or sets a byte value at the specified offset in bytes.
		/// <seealso cref="UnmanagedByteAccessor.this">UnmanagedByteAccessor</seealso>
		/// </summary>
		public UnmanagedByteAccessor Bytes
		{ 
			get
			{
				return byteAccessor;
			}
		}

		#endregion
	}
}
