
/* KDasm.cpp : Defines the entry point for the application.

  ========================================================
  Method 1
	Console Loader for driver which communicates with the
	SysDasm Kernel DLL Disassembly module.
  ========================================================

	There are 2 separate exe/driver source examples in this package.
	The difference lies in how the driver loads the SysDasm.sys
	disassembly module (Linking to the dll module during compiling
	vs. Loading it as an image file).  Other than this difference,
	the functionality and implementation of the 2 examples are identical.

  Method 1:
  =========
	The communicating driver SysDasm_LinkAsDLL.sys is compiled LINKED to
	SysDasm.LIB and its exports. When the driver is loaded by the system,
	the SysDasm.sys module is also loaded as a required	kernel DLL.
	When the driver closes, the DLL is unloaded	by the system.

  Method 2:
  =========
	The communicating driver SysDasm_LoadAsImage.sys handles the
    loading and unloading of the SysDasm.sys kernel Dll module by using
	ZwSetSystemInformation with the classes LoadImage and UnloadImage
	(SYSTEM_LOAD_IMAGE / SYSTEM_UNLOAD_IMAGE).

	Kayaker 2004
*/


#include <windows.h>
#include <stdlib.h>
#include <stdio.h>

#include "DriverControl.h"
#include "Loader.h"
#include "..\common.h"


// Variable length structure returned in array of arrays via MDL
typedef struct _MDL {
	long Address;				// address disassembled
	BYTE OpcodeLength;			// instruction length			
	UCHAR OpcodeBuff[0x10];		// opcodes - up to 16 bytes starting at Address
	UCHAR DisasmText[0x100];	// buffer holding disassembled text
} MDL, *PMDL;


///////////////////////////////////////////////////////
// Display usage
///////////////////////////////////////////////////////

void usage (char *self) {

	printf ("\n This is an example of using the SysDasm Kernel DLL Disassembly module by");
	printf ("\n LINKING to it as a .lib file from another driver.");
	printf ("\n This implementation will disassemble kernel code in one of 2 ways:");

	printf ("\n\n 1. By NT Function Name - any kernel or HAL export resolved by MmGetSystemRoutineAddress.");
	printf ("\n 2. By Address - any valid user or kernel mode instruction address.");

	printf ("\n\n Disassembly will stop at the number of instructions specified (optional),");
	printf ("\n until a RET is reached, or at the default value of 150 instructions.");

	printf ("\n If a FAR JMP (0xEA) instruction is reached, as an indication of a");
	printf ("\n Rootkit hook Detour function, the diverted code will also be disassembled.");
	
	printf ("\n\n Usage:");
	printf ("\n %s df NTFunction_Name <number of instructions>", self);
	printf ("\n %s da Address <number of instructions>", self);
	printf ("\n %s u \t\t\t\t\t - uninstall service", self);

	printf ("\n\n Examples:");
	printf ("\n %s df NtDeviceIoControlFile", self);
	printf ("\n %s df SeAccessCheck 200", self);
	printf ("\n %s da 80464aa0 \n", self);


//	printf ("\n\n Registry keys created during installation of this driver:");
//	printf ("\n HKLM\\SYSTEM\\CurrentControlSet\\Services\\%s", DriverName);
//	printf ("\n HKLM\\SYSTEM\\ControlSet\\Enum\\Root\\LEGACY_%s", DriverName);
//	printf ("\n The service can be uninstalled with:");
//	printf ("\n %s u \t uninstall service\n", self);

	exit (0);
}

///////////////////////////////////////////////////////



///////////////////////////////////////////////////////
///////////////////////////////////////////////////////

BOOL DisasmNTCall (char *function_name, unsigned long address, int num_instructions) {

	IO_IN_BUFFER	pInputBuffer;
	IO_OUT_BUFFER	pOutputBuffer;

	RtlZeroMemory( (void*)&pInputBuffer, sizeof(pInputBuffer) );

	if (address)							// disassemble by address
	{
		pInputBuffer.Address = address;

	} else {								// disassemble by name

		// safe CopyMemory
		memcpy(	pInputBuffer.NtKernelCall,
			(void *) function_name,
			min( strlen(function_name), sizeof(pInputBuffer.NtKernelCall) )
			);
	}

	pInputBuffer.NumToDisasm = num_instructions;
	

	fprintf (stderr, "sending IOCTL_Disassemble...\n");
	if(!DeviceIoControl( hDriver,
		IOCTL_Disassemble,
		(LPVOID)&pInputBuffer,
		sizeof (pInputBuffer),
		(LPVOID)&pOutputBuffer,
		sizeof (pOutputBuffer),
		&BytesReturned,
		NULL))
	{
		fprintf (stderr, "IOCTL request failed\n");
		formatGetLastError ("Error DeviceIoControl:");			
		return FALSE;
	}


	//==================================================
	// Output data from MDL
	//==================================================
	MDL pMdl;

	if (!BytesReturned)
	{
		if (address) {
			fprintf (stderr, "\nCan't disassemble address %X\n", pInputBuffer.Address);
			return FALSE;

		} else {

			fprintf (stderr, "\nCan't disassemble function %s\n", pInputBuffer.NtKernelCall);
		}
		return FALSE;
	}
	
	
	// get MDL buffer address passed via Irp->AssociatedIrp.SystemBuffer
	char * buf = (char *) pOutputBuffer.MDLAddress;

	fprintf (stderr, "\n -------------------------------------------------------\n");

	if (address) {
		fprintf (stderr, "\t Disassembly of %x", pInputBuffer.Address);
	} else {
		fprintf (stderr, "\t Disassembly of %s", pInputBuffer.NtKernelCall);
	}

	fprintf (stderr, "\n -------------------------------------------------------\n");
	fprintf (stderr, "\n#   Address (Len) \t Opcode \t\t  Disassembly\n\n");

	
	int i;
	for (i = 0; i < pOutputBuffer.NumDisasm; i++) 
	{

		// Prepare data
		pMdl.Address = *(long*)buf;
		pMdl.OpcodeLength = *((char *) buf+4);

		memcpy(	pMdl.OpcodeBuff, ((char *) buf+5),
			min( pMdl.OpcodeLength, sizeof(pMdl.OpcodeBuff) ) );

		lstrcpy( (LPTSTR) pMdl.DisasmText, ((char *) buf+5+pMdl.OpcodeLength) );


		// Output Line#, Address
		fprintf (stdout, "%-2d  %p ", i+1, pMdl.Address);
		
		// Output (OpcodeLength)
		fprintf (stdout, " %d  ", pMdl.OpcodeLength);
	
		// Output opcode
		int j;
		for (j = 0; j < (pMdl.OpcodeLength); j++) 
		fprintf (stdout, "%02x", pMdl.OpcodeBuff[j]);	
	
		// format spacing for maximum opcode length
		int k;
		for (k = 0; k < 16-j; k++)
		fprintf (stdout, "  ");	

		// Output disassembly text
		fprintf (stdout, " %s\n", pMdl.DisasmText);	


		//===========================================================
		// The driver has detected if a FAR JMP is reached,
		// assume this is a rootkit hook plus Detour function. 
		//
		//===========================================================

		// EA JMP ptr16:32 Jump far, absolute, address given in operand
		if ( pMdl.OpcodeBuff[0] == 0xEA )
		{
			fprintf (stderr, " ---------------------\n");
			fprintf (stdout, " FAR JMP disassembly\n\n");	
		}

		
		// increment pointer to next array
		int DisasmLength = lstrlen ( (LPCTSTR)pMdl.DisasmText );
		buf = buf + (5 + pMdl.OpcodeLength + DisasmLength+1);	


	}

	fprintf (stdout, "\n");	

	//==================================================

	
	// Cleanup the MDL
	BytesReturned = 0;
	
	fprintf (stderr, "sending IOCTL_MdlCleanup...\n");
	if(!DeviceIoControl( hDriver,
		IOCTL_MdlCleanup,
		NULL,
		0,
		NULL,
		0,
		&BytesReturned,
		NULL))
	{
		fprintf (stderr, "IOCTL request failed\n");
		formatGetLastError ("Error DeviceIoControl:");			
		return FALSE;
	}	

	return TRUE;

}	// DisasmNTCall

///////////////////////////////////////////////////////


///////////////////////////////////////////////////////
// Check driver for validity
///////////////////////////////////////////////////////

BOOL checkFile (PTSTR file) {

	HANDLE hFile = CreateFile (file, GENERIC_READ,
		0, NULL, OPEN_EXISTING, 0, NULL);
	if (hFile == INVALID_HANDLE_VALUE) return FALSE;
	CloseHandle ( hFile );
	return TRUE;
}

///////////////////////////////////////////////////////


///////////////////////////////////////////////////////
// Initialize the driver
///////////////////////////////////////////////////////	
	
BOOL InitDriver() {
		
	if (!checkFile (DriverName)) {

		printf ("\nLoad %s from current directory\n", DriverName);			
		formatGetLastError ("Error CreateFile: ");			
		return FALSE;
	}
		
	fprintf (stderr, "installing service...");
	if (InstallService (DriverName)) {
		fprintf (stderr, "...OK\n");
	}

	fprintf (stderr, "starting driver...");
	if (StartStopService (ServiceName)) {
		fprintf (stderr, "...OK\n");
	}

//	fprintf (stderr, "getting a handle to the driver...");
	if (OpenDevice (ServiceName)) {
//		fprintf (stderr, "...OK\n");
		return TRUE;
	}
}

///////////////////////////////////////////////////////


///////////////////////////////////////////////////////
// Shut the driver down
///////////////////////////////////////////////////////

void CloseDriver() {

	CloseHandle( hDriver );

	fprintf (stderr, "stopping driver...");	
	if (StartStopService (ServiceName) == TRUE)
		fprintf (stderr, "...OK\n");	
	
}

///////////////////////////////////////////////////////


///////////////////////////////////////////////////////
// WinMain
///////////////////////////////////////////////////////

int main(int argc, char* argv[])
{

	PCHAR Name;
	unsigned long Address;
	int NumToDisasm;
	
	if(argc < 2) {
		usage(argv[0]);
		exit(0);
	}

	if(*argv[1]=='u') {							// uninstall service

		printf ("uninstalling service...\n");
		if (UnInstallService (ServiceName) == TRUE)	{
			fprintf (stderr, "...OK\n");
		}	
	return 0;
	}

	if(argc < 3) {
		usage(argv[0]);
		exit(0);
	}
	


	//==================================================
	// Disassemble by NT Function Name or Address
	//==================================================

	if (strcmp((char *)argv[1], "df") == 0)				// by Name
	{			

		// To prevent potential crash by MmGetSystemRoutineAddress
		// don't allow function name to begin with characters 0-9
	
		if( *(char *)(argv[2]) <= 0x39 ) {	// "9"
		
			fprintf(stderr, "Invalid function name: %s\n", argv[2]);
			exit(0);		
		}	

		// OK - initialize Name and Address
		Name = (char *)(argv[2]);
		Address = 0;

	//----------------------------------------------------
		
	} else if (strcmp((char *)argv[1], "da") == 0) {	// by Address

		// Convert string to unsigned long integer using strtoul
		// The string represents an integer number which is parsed 
		// until a non-numeric character is found
		
		char * pEnd;	
		Address = strtoul (
			argv[2],	// string 
			&pEnd,		// returns pointer to where scan ends
			16);		// radix 16 for hexadecimal comparison

		if (Address == -1 || Address < 0x401000) {
			fprintf(stderr, "Invalid address: %s\n", argv[2]);
			exit(0);		
		}	

	} else {
		usage(argv[0]);
		exit(0);		
	}	

	//----------------------------------------------------

	// Check if user specified number of instructions
	// to disassemble, otherwise use a default number

	if (argv[3])
	{
		NumToDisasm = atoi(argv[3]);
			
	} else {

		NumToDisasm = 150;
	}	

	//----------------------------------------------------
	//----------------------------------------------------

	// Initialize the driver		
	if (!InitDriver())
	{
		fprintf(stderr, "Failed to initialize driver \n");
		exit(0);
	}
		
	// Make the DeviceIoControl call
	DisasmNTCall(Name, Address, NumToDisasm);

	// Close the driver handle and stop driver
	CloseDriver();	

	
	//==================================================


	return 0;
}

///////////////////////////////////////////////////////