
/*
  SysDasm_LinkAsDLL.cpp

  Driver which communicates with the SysDasm Kernel DLL Disassembly module.

  Method 1: 
  =========
	The communicating driver SysDasm_LinkAsDLL.sys is compiled LINKED to
	SysDasm.LIB and its exports. When the driver is loaded by the system,
	the SysDasm.sys module is also loaded as a required	kernel DLL.
	When the driver closes, the DLL is unloaded	by the system.

  **Include SysDasm.lib in the project workspace when compiling.**
  
  //////////////////////////////////////////////////////////////////

  In this implementation of SysDasm, we're passing the disassembly
  data back to user mode, using a shared MDL buffer mapped into the
  user address space. The data is passed as variable length structures
  of the form:

	typedef struct _MDL {
		long Address;				// address disassembled
		BYTE OpcodeLength;			// opcode length			
	    UCHAR OpcodeBuff[0x10];		// opcodes - up to 16 bytes starting at Address
		UCHAR DisasmText[0x100];	// buffer holding disassembled text
	} MDL, *PMDL;
  

  The user mode MDL buffer will be filled with arrays of variable length
  MDL structures. To calculate the size of the buffer required, we run
  through the full disassembly once without saving, then a second time
  to fill the newly created MDL buffer.  A pointer to the MDL buffer is
  then passed back to user mode via Irp->AssociatedIrp.SystemBuffer.
	
  //////////////////////////////////////////////////////////////////

  Kayaker 2004

*/

//================================================================

#include "defines.h"
#include "..\common.h"
#include "SysDasm_LinkAsDLL.h"

//================================================================

 const WCHAR DeviceNameBuffer[]		= L"\\Device\\SysDasm_LinkAsDLL";
 const WCHAR SymbolicNameBuffer[]	= L"\\DosDevices\\SysDasm_LinkAsDLL";

 PDEVICE_EXTENSION pDevExt;
 int dwBytesReturned;

 // We pass the disassembly data back to usermode via a shared MDL
 PMDL pMdl;
 PVOID pSharedMemory;
 PVOID pUserAddress;

//================================================================
//================================================================


/////////////////////////////////////////////////////////////////////
//
// DeviceControl
//
/////////////////////////////////////////////////////////////////////
BOOLEAN DeviceControl( IN PVOID pIrpBuffer, IN ULONG pIrpBufferLength, 
        IN ULONG IoControlCode, OUT PIO_STATUS_BLOCK IoStatus )
{

	IoStatus->Status      = STATUS_SUCCESS; // Assume success
    IoStatus->Information = 0;              // Assume nothing returned

    
// Dispatch on IOCTL
switch ( IoControlCode )
{

//################################################################

case IOCTL_Disassemble:

	//////////////////////////////////////////////////////////////////
	// Validate buffer sizes: Input = Output for METHOD_BUFFERED
	//////////////////////////////////////////////////////////////////
	if ( pIrpBufferLength < sizeof(IO_IN_BUFFER) )
	{
		IoStatus->Status = STATUS_INVALID_BUFFER_SIZE;
	    IoStatus->Information = 0;
		return FALSE;		
	}   		
	//////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////
	// Cast Irp->AssociatedIrp.SystemBuffer to our input structure
	//////////////////////////////////////////////////////////////////
	//
	// struct format of data passed to driver
	/*
	typedef struct _IO_IN_BUFFER {
		UCHAR NtKernelCall[0x64];	// name of ntoskrnl or hal function to disassemble
		ULONG Address;				// kernel or user mode address to disassemble
		int NumToDisasm;			// number of instructions to disassemble limit
	} IO_IN_BUFFER, *PIO_IN_BUFFER;
	*/

	IO_IN_BUFFER	pInputBuffer;
	pInputBuffer = *(PIO_IN_BUFFER) pIrpBuffer;
	
	//////////////////////////////////////////////////////////////////
	

	//////////////////////////////////////////////////////////////////
	// Determine if user input is a function name or absolute address
	//////////////////////////////////////////////////////////////////

	PVOID Function_Address;

	if (pInputBuffer.Address)			// if the address field is filled
	{
		Function_Address = (PVOID) pInputBuffer.Address;
	}	

	else if (pInputBuffer.NtKernelCall)		// else it's a function name
	{

		//  Get address of function via MmGetSystemRoutineAddress
		Function_Address = GetAddressFromName((char *)pInputBuffer.NtKernelCall); 
	
		if (!Function_Address)	
		{
			DbgPrint("Cannot find System Routine Address\n");
			IoStatus->Information = 0;		
			return FALSE;	
		}
	} 

	// Store address in our Device Extension
	pDevExt->Address = (ULONG_PTR)Function_Address;

	//////////////////////////////////////////////////////////////////
	

	//////////////////////////////////////////////////////////////////
	// Run through disassembly once without saving to calculate buffer size
	//////////////////////////////////////////////////////////////////

	long LengthOpcode;
	int	i;
	int	len;

	int	szBuffer;
	szBuffer = 0;

	for (i = 0; i < (int) pInputBuffer.NumToDisasm; i++) {

	//===========================================================
	// Call SysDasm Export of the kernel mode disassembly dll
	//===========================================================

		// Check if address is readable, else abort disassembly
	if (!MmIsAddressValid((PVOID)pDevExt->Address))
	{
			DbgPrint("Invalid Address\n");
			IoStatus->Information = 0;		
			return FALSE;	
	}	
		
	// Copy 16 bytes at Address into pDevExt->OpcodeBuff
	RtlCopyMemory ( pDevExt->OpcodeBuff, (long *) pDevExt->Address,
										sizeof pDevExt->OpcodeBuff );	

	LengthOpcode = Disasm( pDevExt->OpcodeBuff,		// IN PUCHAR
						   pDevExt->DisasmText,		// OUT PUCHAR
						   pDevExt->Address);		// IN LONG

	if (LengthOpcode == 0)
	{
		DbgPrint("SYSDASM: ERROR Can't disassemble instruction \n");
		IoStatus->Information = 0;		
		return FALSE;
	}

	//===========================================================

	
	// length of disassembled text
	len = strlen ((char*) (pDevExt->DisasmText));	


	//===========================================================
	// Calculate minimum size of buffer required for MDL structure
	//===========================================================
	szBuffer = szBuffer + 5;				// add 1 dword + 1 byte
	szBuffer = szBuffer + LengthOpcode;		// add length of opcode	
	szBuffer = szBuffer + (len+1);			// add length of text
	

	

	//===========================================================
	// If a FAR JMP (0xEA) is reached, assume this is a rootkit hook
	// plus Detour function. Trace the JMP address and disassemble
	// the diverted code until it returns to the original function.
	//
	// jmp FAR 0x08:0xAAAAAAAA	(Greg Hoglund's MIGBOT example)
	//===========================================================

	LPCTSTR JMPSignature = "jmp ";	// SysDasm returns lowercase disassembly

	// Test if we've reached a JMP instruction
	if( *(long *)(pDevExt->DisasmText) == *(long *)(JMPSignature) )
	{

		// Test if it's a FAR JMP
		// EA - JMP ptr16:32 Jump far, absolute, address given in operand
		if ( pDevExt->OpcodeBuff[0] == 0xEA )
		{
			long JMPAddress;
			JMPAddress = *(long *)(pDevExt->OpcodeBuff+1);	

			// DbgPrint("FAR JMP found to %08x\n", JMPAddress); 

			// Update current disassembly address to the diverted JMP address
			pDevExt->Address = (ULONG_PTR)JMPAddress;
	
			continue;	// break the loop and continue with next disassembly
		}
	}

	//===========================================================


	
	//===========================================================
	// If a RET is reached, end disassembly. This test accounts for
	// opcodes RET (0xC2), RETN (0xC3), RETF (0xCB) and IRET (0xCF)
	//===========================================================
	
	if( *(short *)(pDevExt->DisasmText) == 0x6572			// "re"	
		&& pDevExt->DisasmText[2] == 0x74					// "t"
		|| *(long *)(pDevExt->DisasmText) == 0x74657269 )	// "iret"	

	{
		break;
	}

	//===========================================================

	
	
	//===========================================================
	// Increment address pointer to next valid instruction and repeat
	//===========================================================

	// Next address	
	pDevExt->Address = (pDevExt->Address + LengthOpcode);	

	}	// next FOR

	//////////////////////////////////////////////////////////////////


	
	//////////////////////////////////////////////////////////////////
	//  Build an MDL mapped into user memory
	//////////////////////////////////////////////////////////////////

	pUserAddress = (PVOID) BuildUserMDL(szBuffer);
	if (!pUserAddress)	
	{
		DbgPrint("Cannot create user MDL\n");
		IoStatus->Information = 0;		
		return FALSE;	
	}

	//////////////////////////////////////////////////////////////////

	// Make a working copy of our user buffer pointer
	PVOID MDLBuffer;
	MDLBuffer = pUserAddress;	
	
	

	//////////////////////////////////////////////////////////////////
	// Run through disassembly again, this time filling MDL buffer
	//////////////////////////////////////////////////////////////////

	// Restore original address
	pDevExt->Address = (ULONG_PTR)Function_Address;	

	
	for (i = 0; i < (int) pInputBuffer.NumToDisasm; i++) {

	//===========================================================
	// Call SysDasm Export of the kernel mode disassembly dll
	//===========================================================

	// Copy 16 bytes at Address into pDevExt->OpcodeBuff
	RtlCopyMemory ( pDevExt->OpcodeBuff, (long *) pDevExt->Address,
										sizeof pDevExt->OpcodeBuff );	
		
		
	LengthOpcode = Disasm( pDevExt->OpcodeBuff,
						   pDevExt->DisasmText,
						   pDevExt->Address);

	if (LengthOpcode == 0)
	{
		DbgPrint("SYSDASM: ERROR Can't disassemble instruction \n");
		IoStatus->Information = 0;		
		return FALSE;
	}

	
	// Output disassembly data to debugger window
	//	DbgPrint("Disassembly at Address %X \n", pDevExt->Address);
	//	DbgPrint(" %s \n", pDevExt->DisasmText);
	//	DbgPrint("LengthOpcode = %X \n", LengthOpcode);

	//===========================================================


	//===========================================================
	// Fill MDL buffer with disassembly data
	//===========================================================

	// length of disassembled text
	len = strlen ((char*) (pDevExt->DisasmText));		
	
	
	// Return address of instruction (dword)
	RtlCopyMemory ( MDLBuffer, (long *) &pDevExt->Address, 4);			
	// increment pointer
	MDLBuffer = ((char *) MDLBuffer) + 4;

	// Return opcode length (byte)
	RtlCopyMemory ( MDLBuffer, &LengthOpcode, 1);			
	MDLBuffer = ((char *) MDLBuffer) + 1;
	
	// Return opcode itself (variable length)
	RtlCopyMemory ( MDLBuffer, pDevExt->OpcodeBuff, LengthOpcode);			
	MDLBuffer = ((char *) MDLBuffer) + LengthOpcode;	

	// Return disassembly text (variable length + null terminator)
	RtlCopyMemory ( MDLBuffer, pDevExt->DisasmText, len);			
	MDLBuffer = ((char *) MDLBuffer) + (len+1);	

	//===========================================================
	


	//===========================================================
	// If a FAR JMP (0xEA) is reached, assume this is a rootkit hook
	// plus Detour function. Trace the JMP address and disassemble
	// the diverted code until it returns to the original function.
	//
	// jmp FAR 0x08:0xAAAAAAAA	(Greg Hoglund's MIGBOT example)
	//===========================================================

	LPCTSTR JMPSignature = "jmp ";	// SysDasm returns lowercase disassembly

	// Test if it's a JMP instruction
	if( *(long *)(pDevExt->DisasmText) == *(long *)(JMPSignature) )
	{

		// Test if it's a FAR JMP
		// EA - JMP ptr16:32 Jump far, absolute, address given in operand
		if ( pDevExt->OpcodeBuff[0] == 0xEA )
		{
			long JMPAddress;
			JMPAddress = *(long *)(pDevExt->OpcodeBuff+1);	

			// DbgPrint("FAR JMP found to %08x\n", JMPAddress); 

			// Update current disassembly address to the diverted JMP address
			pDevExt->Address = (ULONG_PTR)JMPAddress;
	
			continue;	// break the loop and continue with next disassembly
		}
	}

	//===========================================================


	
	//===========================================================
	// If a RET is reached, end disassembly. This test accounts for
	// opcodes RET (0xC2), RETN (0xC3), RETF (0xCB) and IRET (0xCF)
	//===========================================================
	
	if( *(short *)(pDevExt->DisasmText) == 0x6572			// "re"	
		&& pDevExt->DisasmText[2] == 0x74					// "t"
		|| *(long *)(pDevExt->DisasmText) == 0x74657269 )	// "iret"	
	{
		i = i + 1;		// this is number of instructions disassembled
		break;
	}

	//===========================================================
	
	
	
	//===========================================================
	// Increment address pointer to next valid instruction and repeat
	//===========================================================	
	
	// Next address	
	pDevExt->Address = (pDevExt->Address + LengthOpcode);	

	}	// next FOR

	//////////////////////////////////////////////////////////////////


	//////////////////////////////////////////////////////////////////
	// Return pointer to MDL to ring 3 via AssociatedIrp.SystemBuffer
	//////////////////////////////////////////////////////////////////

	// Fill output buffer with data to be sent back to usermode
	IO_OUT_BUFFER	pOutputBuffer;
	pOutputBuffer.MDLAddress = pUserAddress;	// pointer to MDL
	pOutputBuffer.NumDisasm = i;	// number of instructions disassembled

	// Copy output buffer to Irp->AssociatedIrp.SystemBuffer
	RtlCopyMemory ( pIrpBuffer, &pOutputBuffer, sizeof pOutputBuffer);			

	dwBytesReturned = sizeof pOutputBuffer;
	
	//////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////	
	// Cleanup MDL and memory later from usermode: IOCTL_MdlCleanup
	//////////////////////////////////////////////////////////////////
	

	//===========================================================
	// Complete the IRP	
	//===========================================================

	IoStatus->Information = dwBytesReturned;
	IoStatus->Status = STATUS_SUCCESS;	
	
	//===========================================================

    break;


//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////
		

//################################################################
 

case IOCTL_MdlCleanup:

	//////////////////////////////////////////////////////////////////	
	// Cleanup our MDL and memory after user mode is done with it
	//////////////////////////////////////////////////////////////////
	
	// Unmap the locked pages from the process address space
	MmUnmapLockedPages (pUserAddress, pMdl);
	// Free the MDL
	IoFreeMdl (pMdl);
	// Free the shared buffer
	ExFreePool(pSharedMemory);

	//////////////////////////////////////////////////////////////////	

	//===========================================================
	// Complete the IRP	
	//===========================================================
	IoStatus->Information = 0;
	IoStatus->Status = STATUS_SUCCESS;
	break;


//################################################################
 

default:
	DbgPrint (("Unknown IRP_MJ_DEVICE_CONTROL\n"));
	IoStatus->Information = 0;
    IoStatus->Status = STATUS_INVALID_DEVICE_REQUEST;
    break;
    }

return TRUE;
}

//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////




/////////////////////////////////////////////////////////////////////
// GetAddressFromName
/////////////////////////////////////////////////////////////////////
PVOID GetAddressFromName (char * function_name)
{

	PVOID Function_Address;
	ANSI_STRING	AnsiString;
	UNICODE_STRING	SystemRoutineName;

	RtlInitAnsiString( &AnsiString, function_name);
	RtlAnsiStringToUnicodeString(&SystemRoutineName, &AnsiString, TRUE);

	Function_Address = (MmGetSystemRoutineAddress(
			&SystemRoutineName)); 

	RtlFreeUnicodeString(&SystemRoutineName);

		if (!Function_Address)
	{			
		return 0;		
	}

	return Function_Address;
}

/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
// BuildUserMDL
//
//		Build an MDL mapped into user memory
//      See http://www.osr.com/ntinsider/2000/sharing_memory.htm
//      Sharing Memory Between Drivers and Applications
/////////////////////////////////////////////////////////////////////

PVOID BuildUserMDL (int szBuffer)
{

	// Allocate memory for MDL
	pSharedMemory = ExAllocatePool(NonPagedPool, szBuffer);
	if (pSharedMemory == NULL)
	{
		DbgPrint("ExAllocatePool failed\n");
		return 0;
	}

	RtlZeroMemory(pSharedMemory, szBuffer);

	// Create an MDL for our page(s) of shared memory
	pMdl = IoAllocateMdl (pSharedMemory, szBuffer, FALSE, FALSE, NULL);	

	// Build the MDL
	MmBuildMdlForNonPagedPool (pMdl);

	// Map into the process address space
    pUserAddress = MmMapLockedPagesSpecifyCache (
							pMdl, UserMode, MmCached,
							NULL, FALSE, NormalPagePriority);

	if (!pUserAddress)
	{
		// free the MDL
		IoFreeMdl (pMdl);
		// free the shared buffer
		ExFreePool(pSharedMemory);		
		
		return 0;
	}

	RtlZeroMemory(pUserAddress, szBuffer);

	return pUserAddress;
}

/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
//
// DispatchControl
// 
/////////////////////////////////////////////////////////////////////

NTSTATUS
DispatchControl (PDEVICE_OBJECT DeviceObject, PIRP Irp)
{

    PIO_STACK_LOCATION      IrpStack;
    PVOID                   inputBuffer;
    ULONG                   inputBufferLength;
    ULONG                   ioControlCode;
	
	
	// Assume success
    Irp->IoStatus.Status      = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;

	// Get pointer to IRP stack location
	IrpStack = IoGetCurrentIrpStackLocation( Irp );

    // Get pointer to input buffer and its length
    inputBuffer             = Irp->AssociatedIrp.SystemBuffer;
    inputBufferLength       = IrpStack->Parameters.DeviceIoControl.InputBufferLength;
    ioControlCode           = IrpStack->Parameters.DeviceIoControl.IoControlCode;	
	
	
	// Dispatch on MajorFunction
	switch (IrpStack->MajorFunction)
	{
		// Handle these two IRPs
		case IRP_MJ_CREATE:
		case IRP_MJ_CLOSE:
		break;	
		
		///////////////////////////////////////////////////////////////

		case IRP_MJ_DEVICE_CONTROL:


			// Handle the GUI IOCTL requests
	        DeviceControl( inputBuffer, inputBufferLength, 
                 ioControlCode, &Irp->IoStatus);
			
			break;

		///////////////////////////////////////////////////////////////
			
		default:
			DbgPrint ("Unknown IRP_MJ_\n");
			Irp -> IoStatus.Status = STATUS_INVALID_DEVICE_REQUEST;
        break;
	
	}


	// Release the IRP 
	IoCompleteRequest(Irp, IO_NO_INCREMENT);

	return STATUS_SUCCESS;
	
}

/////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////
//
// Driver Entry
// 
/////////////////////////////////////////////////////////////////////

EXTERN_C NTSTATUS 
DriverEntry(PDRIVER_OBJECT pDriverObject, PUNICODE_STRING RegistryPath)
{

	NTSTATUS			status;
    UNICODE_STRING		usDeviceName;
    UNICODE_STRING		usSymbolicName;        
	PDEVICE_OBJECT		pDeviceObject = 0;

	RtlInitUnicodeString(&usDeviceName,	DeviceNameBuffer);
	RtlInitUnicodeString(&usSymbolicName, SymbolicNameBuffer);

	status = IoCreateDevice(
		pDriverObject,				// IN PDRIVER_OBJECT 
		sizeof(DEVICE_EXTENSION),	// IN ULONG DeviceExtensionSize
		&usDeviceName,				// IN PUNICODE_STRING 
		FILE_DEVICE_UNKNOWN,		// IN DEVICE_TYPE
		0,							// IN ULONG DeviceCharacteristics
		FALSE,						// IN BOOLEAN Exclusive 
		&pDeviceObject);			// OUT PDEVICE_OBJECT

	if (!NT_SUCCESS(status))
	{
		return status;
	}	

	//------------------------------------------------

	status = IoCreateSymbolicLink(
		&usSymbolicName,		// IN PUNICODE_STRING
		&usDeviceName);			// IN PUNICODE_STRING

	if (!NT_SUCCESS(status))
	{
		IoDeleteDevice (pDriverObject->DeviceObject);
		return status;
	}
	
	//------------------------------------------------

	// Set up dispatch routine entry points for IRP_MJ_Xxx requests

	// IRP's sent by GUI app when opening and closing a handle to the driver.
	pDriverObject->MajorFunction[IRP_MJ_CREATE] = DispatchControl;
	pDriverObject->MajorFunction[IRP_MJ_CLOSE] = DispatchControl;

	// Dispatch routine for DeviceIoControl calls from the GUI app
	pDriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = DispatchControl;

	// Initialize the Device Extension
	pDevExt = (PDEVICE_EXTENSION)pDriverObject->DeviceObject->DeviceExtension; 	
    RtlZeroMemory(pDevExt, sizeof(DEVICE_EXTENSION));

	// Define the DriverUnload procedure
	pDriverObject->DriverUnload = DriverUnload;


	//------------------------------------------------

	return STATUS_SUCCESS;
}

//////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////
//
// DriverUnload
// 
/////////////////////////////////////////////////////////////////////

VOID
DriverUnload (PDRIVER_OBJECT pDriverObject)
{
    UNICODE_STRING		usSymbolicName;        

	// Delete the symbolic link name
	RtlInitUnicodeString(&usSymbolicName, SymbolicNameBuffer);
	IoDeleteSymbolicLink(&usSymbolicName);

	// Delete the device object
	IoDeleteDevice(pDriverObject->DeviceObject);

}

/////////////////////////////////////////////////////////////////////

// EOF