/***************************
	wintruder sdk
	copyright future vision - all rights reserved
	
	this is preliminary information and may be subject to change
***************************/
#pragma once

class Thread;
class Breakpoint;
/*! @brief breakpoint log settings
	@notimplemented
*/
typedef void BPLogData;
/*! @defgroup BPCB_xxx Breakpoint callback return*/
//@{
#define	BPCB_SKIP 0			//!< skip breakpoint
#define BPCB_TAKE 1			//!< take breakpoint
#define BPCB_SKIP_SILENT 2	//!< skip breakpoint, do not add messages to LogList
#define BPCB_TAKE_SILENT 3	//!< take breakpoint, do not add messages to LogList
#define BPCB_COND_ERR 16	//!< used internally 
#define BPCB_COND 17		//!< used internally
//@}

/*!	@brief breakpoint callback prototype
	@param pThread thread in whose context the breakpoint occurred, or NULL on final callback
	@param pBreakpoint the breakpoint that occurred
	@param pCallbackCookie cookie specifed at Breakpoint::SetCallback()
	@returns a value of @ref BPCB_xxx "BPCB_xxx"
	@remarks if @em pThread==NULL the callback function is beeing removed from the breakpoint
		or the breakpoint is killed, @em pCallbackCookie should be uninitialized
	@sa Breakpoint::SetCallback()
*/
typedef int (*pBreakpointCb)(Thread *pThread,Breakpoint* pBreakpoint,void *pCallbackCookie);

/*! @defgroup BPF_xxx Breakpoint flags*/
//@{
#define BPF_STEP	(1<<0)		//!< single stepper
#define BPF_CODE	(1<<1)		//!< code breakpoint
#define BPF_DATA	(1<<2)		//!< data breakpoint
#define BPF_DEFCODE	(1<<8)		//!< BreakpointList::Create() creates code breakpoint
#define BPF_DEFDATA	(1<<9)		//!< BreakpointList::Create() creates data breakpoint
#define BPF_PROPERTY (1<<16)	//!< has property
//@}


#ifdef _USER_SYMBOL
	#include "UserSymbol.h"
	#define _BASE_CLASS UserSymbol
#else
	#include "symbol.h"
	#define _BASE_CLASS Symbol
#endif
/*!	@brief wintruder breakpoint data

	@remarks Identifer() returns @ref SYM_MAKEID(@ref SYMID_BREAKPOINT, @ref BP_xxx "BP_xxx")
**/
class Breakpoint : public _BASE_CLASS
{
public:
/*************
	introducing virtual
*************/
	/*!	@brief get breakpoint flags
		@details get information about the breakpoint (i.e. stepper/code/data) and available property page
		@returns a combination of @ref BPF_xxx "BPF_xxx" values
	**/
	virtual DWORD BreakFlags() const=0;
	/*!	@brief unique breakpoint id
		@return unique breakpoint id
	**/
	virtual DWORD Id()=0;

	virtual BOOL Skip()=0;

	/*! @brief kill the breakpoint
		@return none zero if successful
		@remarks any user defined callback cookie is freed from memory
	**/
	virtual BOOL Kill()=0;

	virtual BOOL Enable()=0;
	virtual BOOL Disable()=0;

	virtual BOOL IsEnabled()=0;

	virtual void SetLog(BPLogData *pLogData)=0;
	virtual BPLogData* GetLog()=0;

	virtual void SetPermanent(BOOL bPermanent)=0;
	virtual BOOL GetPermanent()=0;

	virtual void SetCondition(LPCSTR sCondition,DWORD nDebug)=0;
	virtual LPCSTR GetCondition(DWORD &nDebug)=0;

	virtual void SetSkipCount(DWORD nSkip)=0;
	virtual DWORD GetSkipCount(DWORD &nCurSkip)=0;

	/*! @brief install callback handler for this breakpoint
		@param pCallback pointer to callback function
		@param pCallbackCookie cookie to pass to callback function
		@param pszDescription optional breakpoint/callback description
		@return nonezero if successfull
		@remarks fails if handler already installed
		@remarks pass @em pCallback=NULL to remove callback handler from breakpoint
		@remarks @em pCallback is calles with pThread==NULL, when handler removed or breakpoint killed
		@remarks when a callback handler is used, the permanent settings are ignored, the callback function
			should kill the breakpoint if required
	**/
	virtual BOOL SetCallback(pBreakpointCb pCallback,void* pCallbackCookie,LPCSTR pszDescription=NULL)=0;
	virtual pBreakpointCb GetCallback(LPVOID &pCallbackCookie)=0;
/*************
	constructor
*************/
#ifdef _USER_SYMBOL
	Breakpoint(Symbol *pParent):_BASE_CLASS(pParent){}
#endif
};
#undef _BASE_CLASS


#ifdef _USER_SYMBOL
	#define _BASE_CLASS UserSymbolList
#else
	#define _BASE_CLASS Symbol
#endif
/*!	@brief a list of Breakpoint

	@remarks Identifer() returns @ref SYMID_BREAKPOINTLIST, @ref BP_xxx "BP_xxx"
**/
class BreakpointList : public _BASE_CLASS
{
public:
/*************
	introducing virtual
*************/
	/*!	@brief get breakpoint flags
		@details get information abault available breakpoints (i.e. stepper/code/data), the type of breakpoint
		the Create() function returns and if an advanced property setup page is available
		@returns a combination of @ref BPF_xxx "BPF_xxx" values
	**/
	virtual DWORD BreakFlags() const=0;
	/*!	get a breakpoint by id
		@param id unique breakpoint id
		@return Breakpoint or NULL if id not existing
	**/
	virtual Symbol *ChildById(DWORD id)=0;
	virtual Symbol *Create(DWORDLONG Address,BOOL bPermanent,DWORD nSkip,LPCSTR sCondition,int nDebug,BPLogData *pLogData)=0;
	/*!	@brief create breakpoint ex
		@see CreateBreakpoint.doc
	*/
	virtual Symbol *CreateEx(DWORDLONG Address,...)=0;
	virtual HICON Icon()=0;
/*************
	constructor
*************/
#ifdef _USER_SYMBOL
	BreakpointList(Symbol *pParent):_BASE_CLASS(pParent){}
#endif
};
#undef _BASE_CLASS
