/***************************
	wintruder sdk
	copyright future vision - all rights reserved
	
	this is preliminary information and may be subject to change
***************************/
#pragma once
#include "Tool.h"

/*!	@brief FileLocator callback function
	@param pLocation full path of file, or NULL
	@param param user specified parameter
	@return used in FileLocator::SearchDirectory() only
*/
typedef BOOL (*pFileLocatorCb)(LPCSTR pLocation,void *param);

/*! @defgroup FILELOCATOR_xxx FileLocator flags */
//@{
#define FILELOCATOR_NO_GUI (1<<0)
#define FILELOCATOR_NO_SHOWDELAY (1<<1)
#define FILELOCATOR_NO_RECENT (1<<2)
#define FILELOCATOR_NO_AUTOSEARCH (1<<3)
#define FILELOCATOR_NO_RETURN_FIRST (1<<4)
#define FILELOCATOR_NO_DELAY (1<<5)

#define FILELOCATOR_IGNORE_DEFAULTS (1<<16)
#define FILELOCATOR_IGNORE_FAILED (1<<17)
#define FILELOCATOR_IGNORE_TIMESTAMP (1<<18)
//@}

/*!	@brief Helper for locating files
*/
class FileLocator :
	public Tool
{
public:
/*************
	introducing virtual
*************/
//syncronous
	/*!	@brief %Locate a file w/o gui interface
		@param file file or path to locate, the path is stripped from @em file and used as hint
		@param timestamp required timestamp of file, 0 if explicitely none
		@param hintList semicolon separated path list
		@param flags a combination of @ref FILELOCATOR_xxx "FILELOCATOR_xxx" values
		@param pBuffer buffer for returning location, cannot be NULL
		@return nonezero if successfull
	*/
	virtual BOOL Locate(LPCSTR file,DWORD timestamp,LPCSTR hintList,DWORD flags,LPSTR pBuffer);
//async
	/*!	@brief %Locate a file using callback to window proc
		@param file file or path to locate, the path is stripped from @em file and used as hint
		@param timestamp required timestamp of file, 0 if explicitely none
		@param hintList semicolon separated path list
		@param flags a combination of @ref FILELOCATOR_xxx "FILELOCATOR_xxx" values
		@param pBuffer buffer for returning location, may be NULL
		@param hWnd window handle to receive location
		@param msg message to send to @em hWnd
		@param param user supplied parameter to send to @em hWnd
		@return
			- 0 @em file not found
			- 1 @em pBuffer contains location
			- 2 location of file delayed, callback will be used  (@em wParam may be NULL, if failed to locate file)
			- 3 location returned via callback (in case of @em pBuffer is NULL)
		@remarks callback parameter are
			@code
LRESULT CALLBACK WindowProc(
	HWND  hWnd,		// handle to window
	UINT  msg,		// supplied message
	WPARAM wParam,	// LPCSTR of location, or NULL if failed
	LPARAM lParam	// param
	);
			@endcode
		@sa RemoveDelayed()
	*/
	virtual int Locate(LPCSTR file,DWORD timestamp,LPCSTR hintList,DWORD flags,LPSTR pBuffer,HWND hWnd,UINT msg,void *param=NULL);
	/*!	@brief %Locate a file using callback function
		@param file file or path to locate, the path is stripped from @em file and used as hint
		@param timestamp required timestamp of file, 0 if explicitely none
		@param hintList semicolon separated path list
		@param flags a combination of @ref FILELOCATOR_xxx "FILELOCATOR_xxx" values
		@param pBuffer buffer for returning location, may be NULL
		@param pCallback callback function
		@param param user supplied parameter to send to @em pCallback
		@return
			- 0 @em file not found
			- 1 @em pBuffer contains location
			- 2 location of file delayed, callback will be used  (@em pLocation may be NULL, if failed to locate file)
			- 3 location returned via callback (in case of @em pBuffer is NULL)
		@sa RemoveDelayed()
	*/
	virtual int Locate(LPCSTR file,DWORD timestamp,LPCSTR hintList,DWORD flags,LPSTR pBuffer,pFileLocatorCb pCallback,void *param=NULL);
//search directory
	/*!	@brief search directory for file
		@param file name of file to locate
		@param directory path to search in
		@param pCallback called, if file found, cannot be NULL
		@param pProgress regulary called, may be NULL
		@param param user defined value passed to callback functions
		@return ignore
		@remarks @em pCallback is called, if file found @n
			return zero to stop searching
		@remarks @em pProgress is called regulary, the @em pLocation parameter points to the current search path or may be NULL @n
			return zero to stop searching
	*/
	virtual BOOL SearchDirectory(LPCSTR file,LPCSTR directory,pFileLocatorCb pCallback,pFileLocatorCb pProgress,void *param);
//timestamp
	/*!	@brief get a file's %Timestamp
		@param file full quallified path to file
		@return timestamp of @em file if succeeded, 0 otherwise
		@remarks this function uses a collection of FileFormat symbols added to FileLocator
	*/
	virtual DWORD Timestamp(LPCSTR file);
//remove
	/*! @brief cancel delayed location tasks
		@param pCallback supplied callback function, may be NULL
		@param hWnd supplied window handle, may be NULL
		@remarks to enable destruction of user @em param, %RemoveDelayed calls all delayed callbacks 
			and window procs passing NULL as location
		@remarks %RemoveDelayed should be called before destructing a callback window
	*/
	virtual void RemoveDelayed(pFileLocatorCb pCallback,HWND hWnd);
	/*!	@brief clear list of failed files
	*/
	virtual void ClearFailed();
	/*!	@brief clear list of recent locations
	*/
	virtual void ClearRecent();
	/*!	@brief clear list of search locations
	*/
	virtual void ClearSearch();
	/*!	@brief add search location
		@param path location
		@param bSearch include in search operation
	*/
	virtual void AddSearch(LPCSTR path,bool bSearch);
//gui
	/*!	@brief show list of failed and delayed files
	*/
	virtual void ShowFiles();
};

/*!	@brief Tool returning file properties
*/
class FileFormat : 
	public Tool
{
public:
/*************
	introducing virtual
*************/
	/*!	@brief test if FileFormat handles a specific file
		@param file full path of file
		@return nonezero if file handled
	*/
	virtual BOOL Handles(LPCSTR file) =0;
	/*!	@brief return timestamp of file
		@param file full path of file
		@return timestamp if available, 0 otherwise
	*/
	virtual DWORD Timestamp(LPCSTR file) =0;
};