/***************************
	wintruder sdk
	copyright future vision - all rights reserved
	
	this is preliminary information and may be subject to change
***************************/
#pragma once
#include "symbol.h"

#define ID_USERFRAME_CMD_FIRST	8000
#define ID_USERFRAME_CMD_LAST	8999

/*!	@brief user window callback
	@details optional callback function for windows created by Project::CreateUserWindow()
	@param hWnd window handle
	@param message message send to the window
	@param wParam message specific
	@param lParam message specific
	@param pDefault default window procedure
	@remarks additional user windows will be subclassed, therefore the DefWindowProc() function cannot
	be called when processing any messages. instead the @em pDefault callback must be used.
*/
typedef LRESULT (*USER_WNDPROC)(HWND hWnd,UINT message,WPARAM wParam,LPARAM lParam,WNDPROC pDefault);

/*!	@brief wintruder project
	@remarks Identifer() returns @ref SYM_MAKEID(@ref SYMID_PROJECT, 0)
*/
class Process;
class Thread;
class Debug;
class Frame;
class Project : public Symbol
{
public:
/*************
	introducing virtual
*************/
	virtual BOOL HaveProject();

	virtual void SetDescription(LPCSTR pszDesc);
	virtual LPCSTR Path() const;
//get context
	/*! @brief get current process
		@return currently selected Process or NULL
	*/
	virtual Symbol* GetCurProcess();
	/*! @brief get current thread
		@return currently selected Thread or NULL
	*/
	virtual Symbol* GetCurThread();
	/*! @brief get current debugger
		@return currently selected Debug or NULL
	*/
	virtual Symbol* GetCurDebug();
	/*!	@brief get current frame
		@return currently selected Frame of NULL
	*/
	virtual Symbol* GetCurFrame();
	/*!	@brief get current frame index
		@return index of currently selected Frame or -1 if none
	*/
	virtual int GetCurFrameIndex();
//set context
	/*! @brief set current process
		@param pProcess pointer to new current Process
	*/
	virtual void SetCurProcess(Process *pProcess);
	/*! @brief set current thread
		@param pThread pointer to new current Thread
		@remarks SetProcess if necessary
	*/
	virtual void SetCurThread(Thread *pThread);
	/*! @brief set current debugger
		@param pDebug pointer to new current Debug
	*/
	virtual void SetCurDebug(Debug *pDebug);
	/*!	@brief set current frame
		@param pFrame pointer to new current Frame
		@remarks SetProcess, SetThread and SetDebug if necessary
	*/
	virtual void SetCurFrame(Frame *pFrame);
	/*!	@brief set current frame
		@param nFrame index of new current Frame
	*/
	virtual void SetCurFrameIndex(int nFrame);
//windows
	/*!	@brief create a build in window
		@param id window id, a value of @ref WIN_xxx "WIN_xxx"
		@return HWND of window if created, NULL if creation failed
		@remarks can be called from the main thread only
		@remarks some windows allow one instance only
	*/
	virtual HWND CreateWintruderWindow(UINT id);
	/*!	@brief create a user defined window
		@param id user window id
		@param pViewClass class name of view
		@param nViewStyle style of view
		@param hwndView receives handle of view
		@param pFrameUserWndProc subclass callback of frame
		@param pViewUserWndProc subclass callback of view
		@param hInstance handle of module instance to load resources from or NULL
		@return HWND of frame if created, NULL if failed
		@remarks can be called from the main thread only
		@remarks if @em id is greater @ref WIN_USERFRAME the window becomes identifiable. that is, the window @em id can be used
			in FindWindow(). additionally Wintruder stores and restores window size and toolbar position.
		@par Initialization:
		if @em hInstance is specified and @em id is not 0, the frame is initialized by loading the following 
		resources (MAKEINTRESOURCE(@em id)):
			- icon
			- string (window, toolbar title)
			- context menu
			- toolbar (mfc style)
		@par
		ids of menu items ant toolbar buttons must be in range @ref ID_USERFRAME_CMD_FIRST and @ref ID_USERFRAME_CMD_LAST
		@par
		optionally tool tips and status bar hints are loaded for string ids from @ref ID_USERFRAME_CMD_FIRST to @ref ID_USERFRAME_CMD_LAST.
			the string must be formated as @code "status bar text\ntool tip text" @endcode
	*/
	virtual HWND CreateUserWindow(UINT id,LPCSTR pViewClass,UINT nViewStyle,HWND &hwndView,USER_WNDPROC pFrameUserWndProc,USER_WNDPROC pViewUserWndProc,HINSTANCE hInstance=NULL);
	/*!	@brief find open window
		@param id window id, a value of @ref WIN_xxx "WIN_xxx" or user window id
		@param hWndFindAfter if NULL seach begins at the first window, at the next window otherwise
		@return handle of next window found, NULL if no more windows
	*/
	virtual HWND FindWindow(UINT id,HWND hWndFindAfter=NULL);

	virtual HWND GetWindowView(HWND hWndFrame,int nView);
	/*!	@brief next unused Process id
		@return unique id of any virtual process
	*/
	virtual DWORD NextProcessId();
	/*! @brief add a terminated Process to ProcessList
		@details shortcut for creating terminated copy of @em pProcess
			- copy Process::Path, Process::Parameter(), Process::DefaultPath(), Process::Environment(), Process::InheritDebugging(), Process::ReadOnly(), Process::ReadCache()
			- copy LinkList
		@return new Process symbol, NULL if failed
	*/
	virtual Symbol *AddTerminatedProcess(Process *pProcess);
};
