/***************************
	wintruder sdk
	copyright future vision - all rights reserved
	
	this is preliminary information and may be subject to change
***************************/
/*!	@page PAGE_RegularExpression Regular expression syntax
		<table border="0" cellspacing="1" cellpadding="0" ID="Table1">
			<tr>
				<td colspan="2"><h2><br>
						character sets</h2>
				</td>
			</tr>
			<tr>
				<td>[&nbsp;]</td>
				<td>set of characters whithin brackets</td>
			</tr>
			<tr>
				<td>[^&nbsp;]</td>
				<td>set of characters not within brackets</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						sub pattern</h2>
				</td>
			</tr>
			<tr>
				<td>()</td>
				<td>sub pattern inside brackets</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						references</h2>
				</td>
			</tr>
			<tr>
				<td>\\n</td>
				<td>reference to sub pattern n (1<=n<=9) in order of appearance of opening bracket</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						escapes</h2>
				</td>
			</tr>
			<tr>
				<td>\\xnn</td>
				<td>byte with hex code nn</td>
			</tr>
			<tr>
				<td>\\unnnn</td>
				<td>word with hex code nnnn (big endian)</td>
			</tr>
			<tr>
				<td>\\f</td>
				<td>form feed \\x0c</td>
			</tr>
			<tr>
				<td>\\n</td>
				<td>new line \\x0a</td>
			</tr>
			<tr>
				<td>\\r</td>
				<td>cartrige return \\x0d</td>
			</tr>
			<tr>
				<td>\\t</td>
				<td>tab \\x09</td>
			</tr>
			<tr>
				<td>\\v</td>
				<td>horizontal tab \\x0b</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						metacharacters</h2>
				</td>
			</tr>
			<tr>
				<td>^</td>
				<td>start of data</td>
			</tr>
			<tr>
				<td>$</td>
				<td>end of data</td>
			</tr>
			<tr>
				<td>\\b</td>
				<td>start of word (position bewteen (\\s|^) and \\w)</td>
			</tr>
			<tr>
				<td>\\B</td>
				<td>end of word (position between \\w and (\\s|$))</td>
			</tr>
			<tr>
				<td>\\d</td>
				<td>decimal [0-9]</td>
			</tr>
			<tr>
				<td>\\D</td>
				<td>none decimal [^0-9]</td>
			</tr>
			<tr>
				<td>\\s</td>
				<td>whitespace [ \\x09\\x0a\\x0b\\x0c\\x0d]</td>
			</tr>
			<tr>
				<td>\\S</td>
				<td>none whitespace [^ \\x09\\x0a\\x0b\\x0c\\x0d]</td>
			</tr>
			<tr>
				<td>\\w</td>
				<td>alpha numeric + _ [A-Za-z0-9_]</td>
			</tr>
			<tr>
				<td>\\W</td>
				<td>none alpha numeric or _ [^A-Za-z0-9_]</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						alternatives</h2>
				</td>
			</tr>
			<tr>
				<td>a|b</td>
				<td>matches a or b</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						iterators</h2>
				</td>
			</tr>
			<tr>
				<td>*</td>
				<td>zero or more</td>
			</tr>
			<tr>
				<td>+</td>
				<td>one or more</td>
			</tr>
			<tr>
				<td>?</td>
				<td>zero or one</td>
			</tr>
			<tr>
				<td>{n}</td>
				<td>exactly n times</td>
			</tr>
			<tr>
				<td>{n,}</td>
				<td>at least n times</td>
			</tr>
			<tr>
				<td>{n,m}</td>
				<td>at least n but not more than m times</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						none greedy modifier</h2>
				</td>
			</tr>
			<tr>
				<td>?</td>
				<td>set iterator into none greedy mode</td>
			</tr>
			<tr>
				<td colspan="2"><h2><br>
						extension</h2>
				</td>
			</tr>
			<tr>
				<td colspan="2"><h3><br>
						modifier</h3>
					(applyed locally to current (sub)pattern and children, but not to parents or 
					siblings)</td>
			</tr>
			<tr>
				<td>(?i) or (?+i)</td>
				<td>ignore case</td>
			</tr>
			<tr>
				<td>(?-i)</td>
				<td>case sensitive</td>
			</tr>
			<tr>
				<td>(?g) or (?+g)</td>
				<td>greedy iterator mode</td>
			</tr>
			<tr>
				<td>(?-g)</td>
				<td>none greedy iterator mode</td>
			</tr>
			<tr>
				<td colspan="2"><h3><br>
						alignment</h3>
				</td>
			</tr>
			<tr>
				<td>\\ax</td>
				<td>align search position to x (x-usually a power of 2)</td>
			</tr>
			<tr>
				<td></td>
				<td>
					<table border="0" cellspacing="1" cellpadding="0" ID="Table2">
						<tr>
							<td width="20">x</td>
							<td colspan="3">hex digit align to x (any value 1-fh allowed)</td>
						</tr>
						<tr>
							<td>k</td>
							<td>align to 1k=1024 bytes</td>
							<td>\\Aa</td>
							<td>1024=2^10</td>
						</tr>
						<tr>
							<td>p</td>
							<td>align to page 4k</td>
							<td>\\Ac</td>
							<td>4048=2^12</td>
						</tr>
					</table>
				</td>
			</tr>
			<tr>
				<td>\\Ax</td>
				<td>align search position to 2^x</td>
			</tr>
			<tr>
				<td></td>
				<td>
					<table border="0" cellspacing="1" cellpadding="0" ID="Table3">
						<tr>
							<td width="20">x</td>
							<td colspan="3">hex digit</td>
						</tr>
					</table>
				</td>
			</tr>
			<tr>
				<td colspan="2"><h3><br>
						hexadecimal byte sequences (big endian)</h3>
				</td>
			</tr>
			<tr>
				<td>\\x(n1..n2)</td>
				<td>hex byte</td>
			</tr>
			<tr>
				<td>\\x(n1..n4)</td>
				<td>hex word</td>
			</tr>
			<tr>
				<td>\\x(n1..n8)</td>
				<td>hex dword</td>
			</tr>
			<tr>
				<td>\\x(n1..n16)</td>
				<td>hex quad word</td>
			</tr>
			<tr>
				<td>\\x{n1 n2 n3 ..}</td>
				<td>hex byte string</td>
			</tr>
		</table>
*/

#include "Tool.h"

/*! @brief callback function for advanced search operation
	@param address address of search result or current search postion
	@param pos current search progress in range [0,100]
	@param param user specified callback parameter
	@param subPatternCookie to be used in RegularExpression::SubPattern()
	@remarks used for reporting search results and for progress notifications
	@remarks @em subPatternCookie is valid for reporting search results only, NULL otherwise
	@sa RegularExpression::SearchAll()
*/
typedef BOOL (*pRegExCb)(DWORDLONG address,int pos,void *param,void *subPatternCookie);
/*!	@defgroup REGEXT Regular expression
*/
//@{

/*!	@defgroup REGEX_EXT_xxx Regular expression extension
*/
//@{
#define REGEX_EXT_HEX 1
#define REGEX_EXT_ALIGN 2
#define REGEX_EXT_MODIFIER 4
//@}
/*!	@defgroup REGEX_FLAG_xxx Regular expression flags
*/
//@{
#define REGEX_FLAG_IGNORECASE 1
#define REGEX_FLAG_GREEDY 2
//@}
//@}




class Process;
class RangeArray;
/*!	@brief Tool for searching memory
	@sa @ref PAGE_RegularExpression
*/
class RegularExpression : public Tool
{
public:
/*************
	introducing virtual
*************/
	/*!	@brief parse regular expression
		@details does nothing but checks the syntax of a regular expression
		@param pszRegExString string to parse
		@param ext extensions to be used, a combination of @ref REGEX_EXT_xxx "REGEX_EXT_xxx" values
		@returns 0 if successfully parsed, error code otherwise
	*/
	virtual int Parse(LPCSTR pszRegExString,DWORD ext=-1);
	/*!	@brief map error code to text
		@param nErrorCode error code to map
		@return error description or NULL if no text associated
	*/
	virtual LPCSTR ErrorText(int nErrorCode);
	
	/*!	@brief search binary data
		@param pData pointer to data to find
		@param nData number of byte at @em pData
		@param pProcess Process to search in
		@param start start address of search range
		@param end end address of search range
		@return address of data, or -1 if failed
	*/
	virtual DWORDLONG SearchBinary(BYTE *pData,UINT nData,Process *pProcess,DWORDLONG start,DWORDLONG end);
	/*!	@brief search text string
		@param pszString pointer to text to find
		@param pProcess Process to search in
		@param start start address of search range
		@param end end address of search range
		@param bCaseSensitive nonezero if @em pszString case sensitive
		@return address of data, or -1 if failed
	*/
	virtual DWORDLONG SearchText(LPCSTR pszString,Process *pProcess,DWORDLONG start,DWORDLONG end,BOOL bCaseSensitive);
	/*!	@brief search general regular expression
		@param pszRegExString pointer to expression
		@param pProcess Process to search in
		@param start start address of search range
		@param end end address of search range
		@param flags search flags, a combination of @ref REGEX_FLAG_xxx "REGEX_FLAG_xxx" values
		@param ext regular expression extension, a combination of @ref REGEX_EXT_xxx "REGEX_EXT_xxx" values
		@param nIter maximum number of iterations
		@return address of data, or -1 if failed
	*/
	virtual DWORDLONG Search(LPCSTR pszRegExString,Process *pProcess,DWORDLONG start,DWORDLONG end,DWORD flags=0,DWORD ext=-1,UINT nIter=128);

	/*!	@brief search binary data
		@param pData pointer to data to find
		@param nData number of byte at @em pData
		@param pProcess Process to search in
		@param pSearchRange RangeArray containing search range
		@param pSearchCallback search callback, called for each found item, cannot be NULL
		@param pProgressCallback progress callback, called regulary, may be NULL
		@param param callback cookie
	*/
	virtual void SearchAllBinary(BYTE *pData,UINT nData,Process *pProcess,RangeArray *pSearchRange,pRegExCb pSearchCallback,pRegExCb pProgressCallback,void *param);
	/*!	@brief search text string
		@param pszString pointer to text to find
		@param pProcess Process to search in
		@param pSearchRange RangeArray containing search range
		@param pSearchCallback search callback, called for each found item, cannot be NULL
		@param pProgressCallback progress callback, called regulary, may be NULL
		@param param callback cookie
		@param bCaseSensitive nonezero if @em pszString case sensitive
	*/
	virtual void SearchAllText(LPCSTR pszString,Process *pProcess,RangeArray *pSearchRange,pRegExCb pSearchCallback,pRegExCb pProgressCallback,void *param,BOOL bCaseSensitive);
	/*!	@brief search general regular expression
		@param pszRegExString pointer to expression
		@param pProcess Process to search in
		@param pSearchRange RangeArray containing search range
		@param pSearchCallback search callback, called for each found item, cannot be NULL
		@param pProgressCallback progress callback, called regulary, may be NULL
		@param param callback cookie
		@param flags search flags, a combination of @ref REGEX_FLAG_xxx "REGEX_FLAG_xxx" values
		@param ext regular expression extension, a combination of @ref REGEX_EXT_xxx "REGEX_EXT_xxx" values
		@param nIter maximum number of iterations
		@returns nonzero if @em pszRegExString parsed successfully
	*/
	virtual BOOL SearchAll(LPCSTR pszRegExString,Process *pProcess,RangeArray *pSearchRange,pRegExCb pSearchCallback,pRegExCb pProgressCallback,void *param,DWORD flags=0,DWORD ext=-1,UINT nIter=128);

	/*! @brief match regular expression
		@param pszRegExString pointer to expression
		@param pProcess Process to search in
		@param start start address of data
		@param end end address of data
		@param flags search flags, a combination of @ref REGEX_FLAG_xxx "REGEX_FLAG_xxx" values
		@param ext regular expression extension, a combination of @ref REGEX_EXT_xxx "REGEX_EXT_xxx" values
		@param nIter maximum number of iterations
		@return SymbolList containing sub pattern Range s, NULL if failed
		@remarks if successfull SymbolList contains at least one Range symbol, ranging from @em start to the last address matched by @em pszRegExString.
			furthermore SymbolList contains a Range symbol for each sub pattern ( ) specified in @em pszRegExString.
		@sa @ref PAGE_RegularExpression "Regular expression syntax"
	*/
	virtual Symbol* Match(LPCSTR pszRegExString,Process *pProcess,DWORDLONG start,DWORDLONG end,DWORD flags=0,DWORD ext=-1,UINT nIter=128);
	/*!	@brief get matched sup pattern ranges
		@param subPatternCookie value as passed to @ref pRegExCb
		@return SymbolList containing sub pattern Range s
		@warning valid during calls to @ref pRegExCb only
		@sa Match()
	*/
	virtual Symbol* SubPattern(void *subPatternCookie);
};
