/***************************
	wintruder plugin sample DynamicImports
	copyright future vision - all rights reserved
***************************/
/*! @mainpage Wintruder PlugIn SDK Sample DynamicImports
	@anchor MAIN_DynamicImports
	@section sample_about About DynamicImports
	DynamicImports creates information about dynamically loaded libraries and 
	functions imported by using GetProcAddress.
	@section sample_demonstrates This sample demonstrates
	<ul>
		<li>Basic gui command extension</li>
		<li>Access to Wintruders RootSymbol and Symbol tree</li>
		<li>Register a user defined message type to LogList</li>
		<li>Exception handling</li>
		<li>Breakpoint creation and usage of %Breakpoint callbacks</li>
		<li>Access to the debuggee's execution context and memory</li>
		<ul>
			<li>Enum Frame and read register</li>
			<li>Read some memory of a Process</li>
		</ul>
		<li>Creation of build in symbols and extension of Wintruders Symbol tree</li>
	</ul>
	@section sample_where_start Where to start ?
	<ul>
		<li>@ref DynamicImports.cpp "DynamicImports documentation"</li>
		<li><a href="_dynamic_imports_8cpp-source.html"> DynamicImports source</a></li>
	</ul>
	<ul>
		<li>@ref WintruderSdk.h "Sdk header file"</li>
		<li>@ref PAGE_Hirarchy "Wintruder symbol hirarchy"</li>
	</ul>
**/
#include "StdAfx.h"

#include "resource.h"
#include "MainDef.h"

#include "Process.h"
#include "Thread.h"
#include "Frame.h"
#include "Module.h"

#include "ModuleInfo.h"


#include "Debug.h"
#include "DebugX86Def.h"
#include "Exception.h"
#include "Breakpoint.h"

#include "Config.h"
#include "Log.h"

#include "WindowDef.h"


/*************
	prototypes
*************/
Symbol* HookExport(Process *pProcess,Symbol *pExportList,LPCSTR pszExport,pBreakpointCb pCallback);
int GetProcAddressCB(Thread *pThread,Breakpoint *pBreakpoint,void*);
int LoadLibraryExWCB(Thread *pThread,Breakpoint *pBreakpoint,void*);
BOOL InitReturnCallback(Thread *pThread,UINT nParam,pBreakpointCb pCallback);
int GetProcAddressCB_2(Thread *pThread,Breakpoint *pBreakpoint,void* pData);
int LoadLibraryExWCB_2(Thread *pThread,Breakpoint *pBreakpoint,void* pData);

void AddDynamicImport(Process *pProcess,Module *pBaseModule,Module *pModule,LPCSTR pszName=NULL,WORD ordinal=-1,DWORDLONG nAddress=-1);

void Config(BOOL bStore);
/*************
	global vars
*************/
SymbolPtr<LogList> g_pLogList;
volatile LONG g_nRefCount=0;
BOOL g_bEnabled=TRUE;

HINSTANCE g_hModule;
HWND g_hWndMainFrame;
//wintruder plug-in data
const WPI_Data gWPIData={
	sizeof(WPI_Data),
	PLUGIN_NAME,
	PLUGIN_DESCRIPTION,
	PLUGIN_COPYRIGHT,
	0,
	PLUGIN_FLAGS,
	PLUGIN_DEPENDICES};

/*************
	dllmain and exported functions
*************/
BOOL APIENTRY DllMain(HINSTANCE hModule,DWORD ul_reason_for_call,LPVOID lpReserved)
{
	if(ul_reason_for_call==DLL_PROCESS_ATTACH)
	{
		g_hModule=hModule;
	}
    return TRUE;
}

/*! @brief plugin load callback
	@details
	<ul>
		<li>if the plugin is loaded</li>
		<ul>
			<li>store handle of wintruder window</li>
			<li>get RootSymbol</li>
			<li>get and store pointer to LogList</li>
			<li>load icon of dynamic imports</li>
			<li>register "Dynamic import" to LogList</li>
			<li>load configuration calling Config()</li>
		</ul>
	</ul>
**/
LIBFNC_WPI const WPI_Data* WPI_Load(DWORD version,HWND hWndMainFrame,BOOL bTrigger,void *pReserved)
{
	if(!bTrigger)
	{
		g_hWndMainFrame=hWndMainFrame;

		SymbolPtr<Symbol> pRootSymbol=GetRootSymbol();
		if(pRootSymbol)
		{
		//get config
			Config(FALSE);
		//get log list and register log type
			g_pLogList=pRootSymbol->ChildByIdentifer(SYM_MAKEID(SYMID_LOGLIST,0));
			if(g_pLogList)
			{
				HICON hIcon=LoadIcon(g_hModule,MAKEINTRESOURCE(IDI_DYNIMPORT));
				g_pLogList->RegisterType(LOGTYPE_DYNIMPORT,"Dynamic import",hIcon);
				if(hIcon)
					DestroyIcon(hIcon);

				LPCSTR pszMsg=(g_bEnabled)?PLUGIN_NAME " enabled":PLUGIN_NAME " disabled";
				g_pLogList->AddMessage(pszMsg,LOGTYPE_PLUGIN);
			}
		}
	}
	return (WPI_Data*)&gWPIData;
}
/*! @brief plugin unload callback
	@details just check if PlugIn can be unloaded, if so release pointer to LogList and continue
	<ul>
		<li>store enable state to ConfigFile</li>
		<li>release pointer to LogList</li>
	</ul>
**/
LIBFNC_WPI BOOL WPI_Unload(BOOL bForce)
{
	if(!g_nRefCount||bForce)
	{
		if(g_hWndMainFrame)
		{
			Config(TRUE);
			g_pLogList.Release();
			g_hWndMainFrame=NULL;
		}
		return TRUE;
	}
	return FALSE;
}
/*!	@brief plugin command callback
	@details insert PlugIn commands to Wintruders main menu
	<ul>
		<li>on @ref WPICMD_MENU, just return the menu string</li>
		<li>on @ref WPICMD_MENUCMD, toggle enabled state</li>
		<li>on @ref WPICMD_MENUUPDATE, update menu item according to enabled state</li>
	</ul>
**/
LIBFNC_WPI int WPI_Command(UINT cmdId,UINT winId,HWND hWndFrame,UINT item,char data[2048])
{
	if(winId==WIN_MAINFRAME)
	{
		if(cmdId==WPICMD_MENU)
		{
			strcpy(data,"+0&Hook dynamic imports");
			return 1;
		}
		if(cmdId==WPICMD_MENUCMD)
		{
			ASSERT(item==0);
			g_bEnabled=!g_bEnabled;
		}
		if(cmdId==WPICMD_MENUUPDATE)
		{
			ASSERT(item==0);
			return (g_bEnabled)?MFS_CHECKED:0;
		}
	}
	return 0;
}

/*!	@brief plugin exception callback
	@details if a new @ref DEBUG_X86 process have been initialized, setup hooks
	<ul>
		<li>if in enabled state, and it's the right kind of Exception</li>
		<ul>
			<li>get ModuleList from Process</li>
			<li>get kernel32.dll Module from ModuleList</li>
			<li>get ModuleInfo of kernel32.dll Module</li>
			<li>call ModuleInfo::LoadInformation() to access exports</li>
			<li>get ExportList from ModuleInfo</li>
			<li>call HookExport() to hook GetProcAddress</li>
			<li>call HookExport() to hook LoadLibraryExW</li>
		</ul>
	</ul>
**/
LIBFNC_WPI int WPI_Exception(Interface *pInterface,Process *pProcess,Thread *pThread,Exception *pException,void *pData)
{
	if(	g_bEnabled&&
		pProcess&&
		pException&&
		pException->IsException('WIN','DbEv','Init'))
	{
		SymbolPtr<Symbol> pModuleList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_MODULELIST,0));
		if(pModuleList)
		{
			SymbolPtr<Module> pModule=pModuleList->ChildByName("kernel32.dll");
			if(pModule)
			{
				SymbolPtr<ModuleInfo> pModuleInfo=pModule->GetModuleInfo();
				if(	pModuleInfo&&
					pModuleInfo->LoadInformation(SL_IDENTIFY)&&
					pModuleInfo->LoadInformation(SL_SECTION)&&
					pModuleInfo->LoadInformation(SL_EXPORT))
				{
					SymbolPtr<Symbol> pExportList=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_EXPORTLIST,0));
					if(pExportList)
					{
						SymbolPtr<Breakpoint> pBreak1=HookExport(pProcess,pExportList,"GetProcAddress",GetProcAddressCB);
						if(pBreak1)
						{
							SymbolPtr<Breakpoint> pBreak2=HookExport(pProcess,pExportList,"LoadLibraryExW",LoadLibraryExWCB);
							if(!pBreak2)
								pBreak1->Kill();
						}
					}
				}
			}
		}
	}
	return 0;
}
/*!	@brief handle configuration
	@param bStore TRUE if configuration is to be stored
	@details
	<ul>
		<li>check g_hWndMainFrame and test if plug in initialized</li>
		<li>get RootSymbol</li>
		<li>get ConfigList</li>
		<li>get ConfigFile</li>
		<li>load/store configuration</li>
	</ul>
*/
void Config(BOOL bStore)
{
	if(g_hWndMainFrame)
	{
		SymbolPtr<Symbol> pRootSymbol=GetRootSymbol();
		if(pRootSymbol)
		{
		//get the configfile and load hook state
			SymbolPtr<Symbol> pSym=pRootSymbol->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIGLIST,0));
			if(pSym)
			{
				pSym=pSym->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIG,CONFIG_FILE));
				if(pSym)
				{
					SymbolPtr<ConfigNode> pConfigRoot=pSym->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIG,CONFIG_ROOT));
					if(pConfigRoot)
					{
						pConfigRoot->Config("PlugIn\\" PLUGIN_NAME,"Hook",g_bEnabled,TRUE,bStore);
					}
				}
			}
		}
	}
}

/*!	@brief get grandparent of Symbol
	@param pSym pointer to Symbol
	@returns grandparent of pSym, or NULL if none
*/
Symbol* Parent2(Symbol *pSym)
{
	ASSERT(pSym);
	Symbol *pParent=pSym->Parent();
	if(pParent)
	{
		Symbol *pParent2=pParent->Parent();
		pParent->Release();
		return pParent2;
	}
	return NULL;
}

/*!	@brief hook exported function
	@param pProcess process to attach to
	@param pExportList list of exported functions
	@param pszExport name of exported function to hook
	@param pCallback callback function used with Breakpoint
	@details
	<ul>
		<li>get exported function from pExportList</li>
		<li>get virtual address of Export</li>
		<li>if va is valid</li>
		<ul>
			<li>get x86 software BreakpointList from pProcess</li>
			<li>create the Breakpoint at va of Export</li>
			<li>setup a descriptive name for callback function</li>
			<li>attach the pCallback to the Breakpoint</li>
			<li>increment reference counter</li>
		</ul>
	</ul>
*/
Symbol* HookExport(Process *pProcess,Symbol *pExportList,LPCSTR pszExport,pBreakpointCb pCallback)
{
	SymbolPtr<Symbol> pExport=pExportList->ChildByName(pszExport);
	if(pExport)
	{
		DWORDLONG va=pExport->VA(pProcess);
		if(va!=-1)
		{
			SymbolPtr<BreakpointList> pBreakpointList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86));
			if(pBreakpointList)
			{
				SymbolPtr<Breakpoint> pBreakpoint=pBreakpointList->Create(va,TRUE,0,NULL,0,NULL);
				if(pBreakpoint)
				{
					static char name[15+14+1]="dynamic import ";
					strcpy(name+15,pszExport);

					if(pBreakpoint->SetCallback(pCallback,NULL,name))
					{
						g_nRefCount++;
						return pBreakpoint.AddRef();
					}

					pBreakpoint->Kill();
				}
			}
		}
	}
	return NULL;
}
/*!	@brief callback of GetProcAddress Breakpoint
	@param pThread either Thread where Breakpoint occured or NULL if Breakpoint killed
	@returns return code of @ref pBreakpointCb
	@details
	<ul>
		<li>if Breakpoint is killed, decrease reference counter</li>
		<li>otherwise, setup a callback to the return address</li>
		<li>skip this Breakpoint without notification</li>
	</ul>
*/
int GetProcAddressCB(Thread *pThread,Breakpoint*,void*)
{
	if(!pThread)
	{
		g_nRefCount--;
		return 0;
	}
	InitReturnCallback(pThread,2,GetProcAddressCB_2);
	return BPCB_SKIP_SILENT;
}
/*!	@brief callback of LoadLibraryExW Breakpoint
	@param pThread either Thread where Breakpoint occured or NULL if Breakpoint killed
	@returns return code of @ref pBreakpointCb
	@details
	<ul>
		<li>if Breakpoint is killed, decrease reference counter</li>
		<li>otherwise, setup a callback to the return address</li>
		<li>skip this Breakpoint without notification</li>
	</ul>
*/
int LoadLibraryExWCB(Thread *pThread,Breakpoint*,void*)
{
	if(!pThread)
	{
		g_nRefCount--;
		return 0;
	}
	InitReturnCallback(pThread,3,LoadLibraryExWCB_2);
	return BPCB_SKIP_SILENT;
}
/*!	@brief hook procedures return address and copy some stack parameter
	@param pThread currently interrupted Thread
	@param nParam stack parameter to remember
	@param pCallback callback function to attach to hook
	@returns nonezero if successful
	@details
	<ul>
		<li>get the Process from pThread</li>
		<li>get the x86 indirect BreakpointList from pProcess</li>
		<li>get @ref DEBUG_X86 FrameArray from pThread</li>
		<li>get top level Frame from pFrameArray</li>
		<li>get stack pointer</li>
		<li>allocate parameter buffer and read values from stack</li>
		<li>create the indirect Breakpoint at the procedures return address</li>
		<li>attach pCallback to pBreakpoint</li> 
	</ul>
*/
BOOL InitReturnCallback(Thread *pThread,UINT nParam,pBreakpointCb pCallback)
{
	SymbolPtr<Process> pProcess=Parent2(pThread);
	if(pProcess)
	{
		SymbolPtr<BreakpointList> pBreakIndirectList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86INDIRECT));
		if(pBreakIndirectList)
		{
			SymbolPtr<FrameArray> pFrameArray=pThread->ChildByIdentifer(SYM_MAKEID(SYMID_FRAMEARRAY,DEBUG_X86));
			if(pFrameArray)
			{
				DWORDLONG sp;
				SymbolPtr<Frame> pFrame=pFrameArray->ChildByIndex(0);
				if(	pFrame&&
					pFrame->GetSP(sp))
				{
					DWORD *pData=new DWORD[nParam];
					if(pProcess->GetMemory(sp+4,nParam*sizeof(DWORD),pData))
					{
						SymbolPtr<Breakpoint> pBreak=pBreakIndirectList->CreateEx(sp,(int)4,FALSE,(DWORD)0,(LPCSTR)NULL,(int)0,(BPLogData*)NULL);
						if(pBreak)
						{
							if(pBreak->SetCallback(pCallback,pData))
							{
								g_nRefCount++;
								return TRUE;
							}
						}
					}
					delete []pData;
				}
			}
		}
	}
	return FALSE;
}

/*!	@brief called after GetProcAddress finished
	@param pThread currently interrupted Thread
	@param pBreakpoint the Breakpoint that occured
	@param pData Breakpoint cookie (pointer to copy of original stack parameter)
	@returns return code of @ref pBreakpointCb
	@details
	<ul>
		<li>if the Breakpoint is killed</li>
		<ul>
			<li>decrease reference counter</li>
			<li>delete callback cookie, allocated in InitReturnCallback()</li>
		</ul>
		<li>otherwise</li>
		<ul>
			<li>get Process from pThread</li>
			<li>get FrameArray from pThread</li>
			<li>get top level Frame from pFrameArray</li>
			<li>get contents of eax value</li>
			<li>if GetProcAddress succeeded</li>
			<ul>
				<li>get current Module of pFrame (caller of GetProcAddress)</li>
				<li>get ModuleList from pProcess</li>
				<li>get Module associated to hModule as used at GetProcAddress</li> 
				<li>if not imported by ordinal, read procedure name from pProcess</li>
				<li>AddDynamicImport()</li>
			</ul>
			<li>kill pBreakpoint</li>
			<li>skip this Breakpoint without notification</li>
		</ul>
	</ul>
*/
int GetProcAddressCB_2(Thread *pThread,Breakpoint *pBreakpoint,void* pData)
{
	if(!pThread)
	{
	//breakpoint killed, delete cookie data
		g_nRefCount--;
		delete [](DWORD*)pData;
		return 0;
	}
//get process
	SymbolPtr<Process> pProcess=Parent2(pThread);
	if(pProcess)
	{
	//get threads frame array
		SymbolPtr<FrameArray> pFrameArray=pThread->ChildByIdentifer(SYM_MAKEID(SYMID_FRAMEARRAY,DEBUG_X86));
		if(pFrameArray)
		{
		//get frame and read eax register
			DWORDLONG eax;
			SymbolPtr<Frame> pFrame=pFrameArray->ChildByIndex(0);
			if(	pFrame&&
				pFrame->GetRegister(debugX86REG_EAX,eax))
			{
			//check if GetProcAddress successful
				if(eax)
				{
					SymbolPtr<Module> pCallerModule=pFrame->GetModule();
					if(pCallerModule)
					{
					//referenced module
						SymbolPtr<Symbol> pModuleList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_MODULELIST,0));
						if(pModuleList)
						{
							SymbolPtr<Module> pModule=pModuleList->ChildByVA(((DWORD*)pData)[0],pProcess);
							if(pModule)
							{
								DWORD vaName=((DWORD*)pData)[1];
								if(vaName>>16)
								{
								//get function name
									char buf[MAX_PATH];
									*buf=0;
									for(int k=0;k<sizeof(buf)-1;k++)
									{
										if(	!pProcess->GetMemory(vaName+k,1,&buf[k])||
											!buf[k])
											break;
									}
									buf[k]=0;
								//add dynamic import
									ASSERT(*buf);
									AddDynamicImport(pProcess,pCallerModule,pModule,buf);
								}else
								{
								//add by ordinal
									ASSERT(LOWORD(vaName)!=0xffff);
									AddDynamicImport(pProcess,pCallerModule,pModule,NULL,LOWORD(vaName));
								}
							}//module
						}//module list
					}//caller module
				}//successful function
			}//frame & register
		}//frame array
	}//process
//kill the intermediate breakpoint
	pBreakpoint->Kill();
	return BPCB_SKIP_SILENT;
}

/*!	@brief called after LoadLibraryExW finished
	@param pThread currently interrupted Thread
	@param pBreakpoint the Breakpoint that occured
	@param pData Breakpoint cookie (pointer to copy of original stack parameter)
	@returns return code of @ref pBreakpointCb
	@details
	<ul>
		<li>if the Breakpoint is killed</li>
		<ul>
			<li>decrease reference counter</li>
			<li>delete callback cookie, allocated in InitReturnCallback()</li>
		</ul>
		<li>otherwise</li>
		<ul>
			<li>get Process from pThread</li>
			<li>get FrameArray from pThread</li>
			<li>get top level Frame from pFrameArray</li>
			<li>get contents of eax value</li>
			<li>if LoadLibraryExW succeeded</li>
			<ul>
				<li>get ModuleList from pProcess</li>
				<li>get Module of kernel32.dll and ntdll.dll from pModuleList</li>
				<li>enumerate frames of pFrameArray</li>
				<ul>
					<li>get Module from pFrame</li>
					<li>if pModule if this is another Module than kernel32 and ntdll</li> 
					<ul>
						<li>get the Module imported from pModuleList</li>
						<li>AddDynamicImport()</li>
					</ul>
				</ul>
			</ul>
			<li>kill pBreakpoint</li>
			<li>skip this Breakpoint without notification</li>
		</ul>
	</ul>
*/
int LoadLibraryExWCB_2(Thread *pThread,Breakpoint *pBreakpoint,void* pData)
{
	if(!pThread)
	{
	//breakpoint killed, delete cookie data
		g_nRefCount--;
		delete [](DWORD*)pData;
		return 0;
	}
//get process
	SymbolPtr<Process> pProcess=Parent2(pThread);
	if(pProcess)
	{
	//get threads frame array
		SymbolPtr<FrameArray> pFrameArray=pThread->ChildByIdentifer(SYM_MAKEID(SYMID_FRAMEARRAY,DEBUG_X86));
		if(pFrameArray)
		{
		//get top level frame, and read eax register
			DWORDLONG eax;
			SymbolPtr<Frame> pFrame=pFrameArray->ChildByIndex(0);
			if(	pFrame&&
				pFrame->GetRegister(debugX86REG_EAX,eax))
			{
			//check if LoadLibraryExW successful
				if(eax)
				{
				//find calling module (different kernel32, ntdll)
					SymbolPtr<Symbol> pModuleList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_MODULELIST,0));
					if(pModuleList)
					{
						SymbolPtr<Symbol> pModuleKernel=pModuleList->ChildByName("kernel32.dll");
						SymbolPtr<Symbol> pModuleNtdll=pModuleList->ChildByName("ntdll.dll");
						SymbolPtr<Module> pModule;
					//...enumerate frames
						for(int i=0;;i++)
						{
							pFrame=pFrameArray->ChildByIndex(i);
							if(!pFrame)
								break;
						//get frames module
							pModule=pFrame->GetModule();
							if(	pModule&&
								pModule!=pModuleKernel&&
								pModule!=pModuleNtdll)
							{
							//get the imported module
								SymbolPtr<Module> pImportedModule=pModuleList->ChildByVA(eax,pProcess);
								if(pImportedModule)
								{
								//add module to dynamic imports
									AddDynamicImport(pProcess,pModule,pImportedModule);
								}
#ifdef _DEBUG
								else
								{
								//something went wrong, read the name of the library
									DWORD vaName=((DWORD*)pData)[0];
									WCHAR buf[MAX_PATH];
									*buf=0;
									for(int k=0;;k++)
									{
										if(	!pProcess->GetMemory(vaName+k*2,2,&buf[k])||
											!buf[k])
											break;
									}
									ASSERT(FALSE);
								}
#endif
								break;
							}//different module
						}//enum frames
					}//module list
				}//successful function
			}//frame & register
		}//frame array
	}//process
//kill the intermediate breakpoint
	pBreakpoint->Kill();
	return BPCB_SKIP_SILENT;
}

/*!	@brief add dynamic Import
	@param pProcess current Process
	@param pBaseModule importer Module
	@param pModule importee Module
	@param pszName optional name of import
	@param ordinal optional ordinal of import
	@param nAddress address of Import
	@details
	<ul>
		<li>get ModuleInfo from pBaseModule</li>
		<li>load imports from pModuleInfo</li>
		<li>get ImportList from pModuleInfo or generate if not existing</li>
		<li>enumerate ImportModule of pImportList</li>
		<ul>
			<li>check if dynamic import of pModule already present</li>
			<li>otherwise create it</li>
		</ul>
		<li>if pzsName or ordinal have been supplied</li>
		<ul>
			<li>calulate rva of Import</li>
			<li>if necessary genereat oridinal name</li>
			<li>try to get Import from pImportList</li>
			<li>create Import if not present</li>
			<li>generate some output to LogList</li>
			<ul>
				<li>generate string</li>
				<li>add to LogList</li>
			</ul>
		</ul>
	</ul>
*/
void AddDynamicImport(Process *pProcess,Module *pBaseModule,Module *pModule,LPCSTR pszName,WORD ordinal,DWORDLONG nAddress)
{
	SymbolPtr<ModuleInfo> pModuleInfo=pBaseModule->GetModuleInfo();
	if(	pModuleInfo&&
		pModuleInfo->LoadInformation(SL_IDENTIFY)&&
		pModuleInfo->LoadInformation(SL_IMPORT))
	{
	//get imports
		SymbolPtr<Symbol> pImportList=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_IMPORTLIST,0));
		if(!pImportList)
		{
		//not existing -> create it
			pImportList=CreateSymbol(pModuleInfo,SYM_MAKEID(SYMID_IMPORTLIST,0));
			if(pImportList)
			{
				if(!pModuleInfo->AddChild(pImportList))
					pImportList.Release();
			}
		}
		if(pImportList)
		{
		//find import module entry
			SymbolPtr<Symbol> pImportModule;
			POSITION pos=pImportList->FirstChildPosition();
			while(pos)
			{
				pImportModule=pImportList->NextChild(pos);
				if(	!stricmp(pImportModule->Name(),pModule->Name())&&
					SYM_SUBID(pImportModule->Identifer())==IMPORT_DYNAMIC)
					break;
				pImportModule.Release();
			}
			if(!pImportModule)
			{
			//not found -> create it
				pImportModule=CreateSymbol(pImportList,SYM_MAKEID(SYMID_IMPORTMODULE,IMPORT_DYNAMIC),pModule->Name(),(DWORD)-1,(DWORD)4);
				if(pImportModule)
				{
					if(!pImportList->AddChild(pImportModule))
						pImportModule.Release();
				}
			}
			if(pImportModule)
			{
				if(pszName||ordinal!=(WORD)-1)
				{
				//find that imported function
					DWORDLONG nAddressRVA=nAddress;
					if(nAddress!=-1)
						nAddressRVA-=pBaseModule->VA(pProcess);

					char sOrdinal[8];
					if(!pszName)
					{
					//generate ordinal name
						sprintf(sOrdinal,"#%u",ordinal);
						pszName=sOrdinal;
					}

					SymbolPtr<Symbol> pImportEntry=pImportModule->ChildByName(pszName);
					if(!pImportEntry)
					{
					//not found -> create it
						pImportEntry=CreateSymbol(pImportModule,SYM_MAKEID(SYMID_IMPORT,IMPORT_DYNAMIC),pszName,ordinal,(WORD)nAddressRVA);
						if(pImportEntry)
							pImportModule->AddChild(pImportEntry);
					}
				}//Name
				if(g_pLogList)
				{
				//some output
					int len=6;
					len+=strlen(pBaseModule->Name());
					len+=strlen(pImportModule->Name());
					if(pszName)
						len+=strlen(pszName);
					char *buf=new char[len];
					strcpy(buf,pBaseModule->Name());
					strcat(buf," -> ");
					strcat(buf,pImportModule->Name());
					if(pszName)
					{
						strcat(buf,"!");
						strcat(buf,pszName);
					}
					g_pLogList->AddMessage(buf,LOGTYPE_DYNIMPORT);
					delete []buf;
				}
			}//pImportModule
		}//import list
	}//module info
}
