/***************************
	wintruder plugin sample FunctionBreakpoints
	copyright future vision - all rights reserved
***************************/
#include "StdAfx.h"

#include "FunctionBreakpoints.h"
#include "BreakpointX86Function.h"

#include "SyncState.h"
#include "BroadcastDef.h"

#include "Log.h"

#include "Process.h"
#include "Thread.h"
#include "Frame.h"

#include "UserId.h"

//#define _SUSPICIOUS

/*************
	BreakpointListX86Function
*************/
BreakpointListX86Function::BreakpointListX86Function(Symbol *pParent)
	:BreakpointList(pParent)
{
	m_nId=0;
}
BreakpointListX86Function::~BreakpointListX86Function()
{
	ResetChildren();
}
DWORDLONG BreakpointListX86Function::Identifer() const
{
	return SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86FNC);
}
LPCSTR BreakpointListX86Function::Name()
{
	return "BreakpointList";
}
LPCSTR BreakpointListX86Function::Type()
{
	return "function";
}
BOOL BreakpointListX86Function::AddChild(Symbol *pSymbol)
{
	ASSERT(pSymbol&&pSymbol->Identifer()==SYM_MAKEID(SYMID_BREAKPOINT,BP_X86FNC));
	if(	pSymbol&&
		pSymbol->Identifer()==SYM_MAKEID(SYMID_BREAKPOINT,BP_X86FNC))
	{
		if(UserSymbolList::AddChild(pSymbol))
		{
			g_pSyncState->Broadcast(BROADCAST_SYMBOL,H_BREAKPOINT_ADD,pSymbol);
			return TRUE;
		}
	}
	return FALSE;
}
BOOL BreakpointListX86Function::RemoveChild(Symbol *pSymbol)
{
	ASSERT(pSymbol);
	g_pSyncState->Broadcast(BROADCAST_SYMBOL,H_BREAKPOINT_REMOVE,pSymbol);
	return UserSymbolList::RemoveChild(pSymbol);
}
DWORD BreakpointListX86Function::BreakFlags() const
{
	if(g_bGuiEnabled)
		return BPF_CODE|BPF_DEFCODE;
	return BPF_CODE;
}
Symbol *BreakpointListX86Function::ChildById(DWORD id)
{
	SymbolPtr<Breakpoint> pBreakpoint;
	POSITION pos=FirstChildPosition();
	while(pos)
	{
		pBreakpoint=NextChild(pos);
		if(pBreakpoint->Id()==id)
			return pBreakpoint.AddRef();
	}
	return NULL;
}
Symbol *BreakpointListX86Function::Create(DWORDLONG Address,BOOL bPermanent,DWORD nSkip,LPCSTR sCondition,int nDebug,BPLogData *pLogData)
{
	EnterCriticalSection(&g_csBreakpointList);

	SymbolPtr<BreakpointX86Function> pData=ChildByVA(Address,NULL);
	if(pData)
	{
		pData->SetCallbackEx(NULL,NULL);
	}else
	{
		pData=new BreakpointX86Function(this,m_nId,Address);
		m_nId++;
		AddChild(pData);
	}
	pData->SetLog(pLogData);
	pData->SetPermanent(bPermanent);
	pData->SetCondition(sCondition,nDebug);
	pData->SetSkipCount(nSkip);

	LeaveCriticalSection(&g_csBreakpointList);

	if(pData->Enable())
		return pData.AddRef();
	pData->Kill();
	return NULL;
}
Symbol *BreakpointListX86Function::CreateEx(DWORDLONG Address,...)
{
	va_list marker;
	va_start(marker,Address);
	pBreakpointX86FunctionCb pCallback=va_arg(marker,pBreakpointX86FunctionCb);
	void* pCallbackCookie=va_arg(marker,void*);
	LPCSTR pszDescription=va_arg(marker,LPCSTR);
	va_end(marker);

	ASSERT(pCallback);
	SymbolPtr<BreakpointX86Function> pBreakpoint=Create(Address,TRUE,0,NULL,0,NULL);
	if(pBreakpoint)
	{
		if(pBreakpoint->SetCallbackEx(pCallback,pCallbackCookie,pszDescription))
			return pBreakpoint.AddRef();
		pBreakpoint->Kill();
	}
	return NULL;
}
HICON BreakpointListX86Function::Icon()
{
	return NULL;
}
/*************
	BreakpointX86Function
*************/
BreakpointX86Function::BreakpointX86Function(Symbol *pParent,DWORD nId,DWORDLONG nAddress)
	:Breakpoint(pParent)
{
	m_nAddress=nAddress;
	m_nId=nId;
	m_bPermanent=TRUE;
	m_pszDescription=NULL;
	m_pCallback=NULL;
	m_pCallbackCookie=NULL;
	m_pszCondition=NULL;
	m_pBreakpointX86=NULL;
}
BreakpointX86Function::~BreakpointX86Function()
{
	Disable();
	ASSERT(!m_pBreakpointX86);
	if(m_pszDescription)
		delete []m_pszDescription;
	if(m_pszCondition)
		delete []m_pszCondition;
	ASSERT(!m_pCallback);
}
DWORDLONG BreakpointX86Function::Identifer() const
{
	return SYM_MAKEID(SYMID_BREAKPOINT,BP_X86FNC);
}
LPCSTR BreakpointX86Function::Name()
{
	return "Breakpoint";
}
LPCSTR BreakpointX86Function::Type()
{
	return "function";
}
LPCSTR BreakpointX86Function::Description()
{
	return m_pszDescription;
}
DWORDLONG BreakpointX86Function::VA(Process*)
{
	return m_nAddress;
}
DWORDLONG BreakpointX86Function::Size()
{
	return 1;
}

DWORD BreakpointX86Function::BreakFlags() const
{
	return BPF_CODE;
}
DWORD BreakpointX86Function::Id()
{
	return m_nId;
}
BOOL BreakpointX86Function::Skip()
{
	ASSERT(FALSE);
	return FALSE;
}
BOOL BreakpointX86Function::Kill()
{
	EnterCriticalSection(&g_csBreakpointList);

	Disable();

	SetCallbackEx(NULL,NULL);

	SymbolPtr<Symbol> pList=Parent();
	if(pList)
		pList->RemoveChild(this);

	LeaveCriticalSection(&g_csBreakpointList);
	return TRUE;
}

BOOL BreakpointX86Function::Enable()
{
	if(!m_pBreakpointX86)
	{
		SymbolPtr<Symbol> pBreakpointList=Parent();
		if(pBreakpointList)
		{
			SymbolPtr<Symbol> pProcess=pBreakpointList->Parent();
			if(pProcess)
			{
				SymbolPtr<BreakpointList> pBreakX86List=pProcess->ChildByIdentifer(
						SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86));
				if(pBreakX86List)
				{
					m_pBreakpointX86=(Breakpoint*)pBreakX86List->Create(m_nAddress,m_bPermanent,m_nSkipCount,m_pszCondition,m_nDebug,NULL);
					if(m_pBreakpointX86)
					{
						if(m_pBreakpointX86->SetCallback(BreakpointX68Cb,this,"FunctionBreakpoint"))
							return TRUE;
						m_pBreakpointX86->Kill();
					}
				}
			}
		}
		return FALSE;
	}
	return m_pBreakpointX86->Enable();
}
BOOL BreakpointX86Function::Disable()
{
	if(m_pBreakpointX86)
	{
		m_pBreakpointX86->Kill();
		m_pBreakpointX86->Release();
		m_pBreakpointX86=NULL;
		return TRUE;
	}
	return FALSE;
}
BOOL BreakpointX86Function::IsEnabled()
{
	if(m_pBreakpointX86)
		return m_pBreakpointX86->IsEnabled();
	return FALSE;
}
void BreakpointX86Function::SetLog(BPLogData *)
{
}
BPLogData *BreakpointX86Function::GetLog()
{
	return NULL;
}
void BreakpointX86Function::SetPermanent(BOOL bPermanent)
{
	m_bPermanent=bPermanent;
	if(m_pBreakpointX86)
		m_pBreakpointX86->SetPermanent(m_bPermanent);
}
BOOL BreakpointX86Function::GetPermanent()
{
	return m_bPermanent;
}
void BreakpointX86Function::SetCondition(LPCSTR sCondition,DWORD nDebug)
{
	m_nDebug=nDebug;
	if(m_pszCondition)
	{
		delete []m_pszCondition;
		m_pszCondition=NULL;
	}
	if(sCondition)
	{
		int len=strlen(sCondition)+1;
		m_pszCondition=new char[len];
		memcpy(m_pszCondition,sCondition,len);
	}
	if(m_pBreakpointX86)
		m_pBreakpointX86->SetCondition(m_pszCondition,m_nDebug);
}
LPCSTR BreakpointX86Function::GetCondition(DWORD &nDebug)
{
	nDebug=m_nDebug;
	return m_pszCondition;
}
void BreakpointX86Function::SetSkipCount(DWORD nSkip)
{
	m_nSkipCount=nSkip;
	if(m_pBreakpointX86)
		m_pBreakpointX86->SetSkipCount(m_nSkipCount);
}
DWORD BreakpointX86Function::GetSkipCount(DWORD &nCurSkip)
{
	if(m_pBreakpointX86)
		return m_pBreakpointX86->GetSkipCount(nCurSkip);
	nCurSkip=m_nSkipCount;
	return m_nSkipCount;
}
BOOL BreakpointX86Function::SetCallback(pBreakpointCb pCallback,void *pCallbackCookie,LPCSTR pszDescription)
{
	ASSERT(FALSE);
	return NULL;
}
BOOL BreakpointX86Function::SetCallbackEx(pBreakpointX86FunctionCb pCallback,void *pCallbackCookie,LPCSTR pszDescription)
{
	if(!pCallback)
	{
		if(m_pCallback)
		{
			pBreakpointX86FunctionCb pTemp=m_pCallback;
			m_pCallback=NULL;
			pTemp(NULL,this,m_pCallbackCookie,BPFNC_REMOVE);
		}
		pCallbackCookie=NULL;
	}
	if(m_pCallback)
		return FALSE;
	m_pCallback=pCallback;
	m_pCallbackCookie=pCallbackCookie;
	if(m_pszDescription)
	{
		delete []m_pszDescription;
		m_pszDescription=NULL;
	}
	if(pszDescription)
	{
		int len=strlen(pszDescription)+1;
		m_pszDescription=new char[len];
		memcpy(m_pszDescription,pszDescription,len);
	}
	return TRUE;
}
pBreakpointCb BreakpointX86Function::GetCallback(LPVOID &pCallbackCookie)
{
	return NULL;
}
pBreakpointX86FunctionCb BreakpointX86Function::GetCallbackEx(LPVOID &pCallbackCookie)
{
	pCallbackCookie=m_pCallbackCookie;
	return m_pCallback;
}

int BreakpointX86Function::BreakpointX68Cb(Thread *pThread,Breakpoint* pBreakpoint,void* pCookie)
{
	return ((BreakpointX86Function*)pCookie)->OnBreakpointX68(pThread,pBreakpoint);
}
int BreakpointX86Function::BreakpointIndirectCb(Thread *pThread,Breakpoint* pBreakpoint,void* pCookie)
{
	return ((BreakpointX86Function*)pCookie)->OnBreakpointIndirect(pThread,pBreakpoint);
}
int BreakpointX86Function::OnBreakpointX68(Thread *pThread,Breakpoint *pBreakpoint)
{
	if(!pThread)
		return 0;

//function entered
	int nReason=BPFNC_ERROR;
	SymbolPtr<Symbol> pSymbol=pThread->Parent();
	if(pSymbol)
	{
		SymbolPtr<Process> pProcess=pSymbol->Parent();
		if(pProcess)
		{
			SymbolPtr<BreakpointList> pBpIndirectList=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86INDIRECT));
			if(pBpIndirectList)
			{
				SymbolPtr<FrameArray> pFrameArray=pThread->ChildByIdentifer(SYM_MAKEID(SYMID_FRAMEARRAY,DEBUG_X86));
				if(pFrameArray)
				{
					DWORDLONG sp;
					SymbolPtr<Frame> pFrame=pFrameArray->ChildByIndex(0);
					if(	pFrame&&
						pFrame->GetSP(sp))
					{
						DWORD nAddress;
						if(pProcess->GetMemory(sp,sizeof(nAddress),&nAddress))
						{
							SymbolPtr<Breakpoint> pBreakIndirect=pBpIndirectList->CreateEx(
									sp,(int)4,FALSE,(DWORD)0,(LPCSTR)NULL,(int)0,(BPLogData*)NULL);
							if(pBreakIndirect)
							{
								if(pBreakIndirect->SetCallback(BreakpointIndirectCb,this,"FunctionBreakpoint ret"))
								{
									nReason=BPFNC_ENTER;
#ifdef _SUSPICIOUS
								//sanity check
									DWORDLONG ip;
									pFrame=pFrameArray->ChildByIndex(1);
									if(	!pFrame||
										!pFrame->GetIP(ip)||
										ip!=nAddress)
									{
										ASSERT(FALSE);
										nReason=BPFNC_SUSPICIOUS;
									}
#endif
								}else
									pBreakIndirect->Kill();
							}
						}
					}
				}
			}
		}
	}
	return OnBreakpoint(pThread,nReason);	
}
int BreakpointX86Function::OnBreakpointIndirect(Thread *pThread,Breakpoint *pBreakpoint)
{
	if(!pThread)
		return 0;
//function returned
	pBreakpoint->Kill();
	return OnBreakpoint(pThread,BPFNC_RETURN);
}
int BreakpointX86Function::OnBreakpoint(Thread *pThread,int nReason)
{
	int nReturn;
	if(m_pCallback)
		nReturn=m_pCallback(pThread,this,m_pCallbackCookie,nReason);
	else
		nReturn=BPCB_TAKE;
	if(nReturn==BPCB_TAKE)
	{
		SymbolPtr<Process> pProcess=pThread->Parent();
		if(pProcess)
			pProcess=pProcess->Parent();

		char msg[20+16+5+8 +5+8 +1+MAX_PATH+3+10];

		sprintf(msg,"Function breakpoint %8.8I64X TID %8.8X",m_nAddress,pThread->Id());
		if(pProcess)
		{
			sprintf(msg+strlen(msg)," PID %8.8X %s",pProcess->Id(),pProcess->Name());
		}

		LPCSTR pszState;
		switch(nReason)
		{
		case BPFNC_ENTER:
			pszState="enter";break;
		case BPFNC_RETURN:
			pszState="return";break;
		case BPFNC_ERROR:
			pszState="error";break;
		case BPFNC_SUSPICIOUS:
			pszState="suspicious";break;
		case BPFNC_REMOVE:
			pszState="remove";break;
		default:
			ASSERT(FALSE);
			pszState=NULL;
		}
		if(pszState)
		{
			strcat(msg,"\r\n\t");
			strcat(msg,pszState);
		}
		g_pLogList->AddMessage(msg,LOGTYPE_BREAKPOINT|LOGEVENT_EVENT);

		g_pSyncState->Broadcast(BROADCAST_SYMBOL,H_BREAKPOINT,this);

		if(	nReason==BPFNC_RETURN&&
			!m_bPermanent)
		{
			Kill();
		}
	}
	return nReturn;
}
