/***************************
	wintruder plugin sample FunctionBreakpoints
	copyright future vision - all rights reserved
***************************/
/*! @mainpage Wintruder PlugIn SDK Sample FunctionBreakpoints
	@anchor MAIN_FunctionBreakpoints
	@section sample_about About FunctionBreakpoints
	FunctionBreakpoints 
	@section sample_demonstrates This sample demonstrates
	<ul>
		<li>Basic gui command extension</li>
		<li>Access to Wintruders RootSymbol and Symbol tree</li>
		<li>Usage of Broadcast callback</li>
		<li>Respond to Broadcast command</li>
		<li>Breakpoint creation and usage of %Breakpoint callbacks</li>
		<li>Access to the debuggee's execution context and memory</li>
		<ul>
			<li>Enum Frame and read register</li>
			<li>Read some memory of a Process</li>
		</ul>
		<li>Creation of user defined symbols and extension of Wintruders Symbol tree</li>
	</ul>
	@section sample_where_start Where to start ?
	<ul>
		<li>@ref FunctionBreakpoints.cpp "FunctionBreakpoints documentation"</li>
		<li><a href="_function_breakpoints_8cpp-source.html">FunctionBreakpoints source</a></li>
		<li>BreakpointX86Funcion class</li>
	</ul>
	<ul>
		<li>@ref WintruderSdk.h "Sdk header file"</li>
		<li>@ref PAGE_Hirarchy "Wintruder symbol hirarchy"</li>
	</ul>
**/
#include "StdAfx.h"

#include "resource.h"
#include "MainDef.h"

#include "BreakpointX86Function.h"

#include "Process.h"
#include "Thread.h"

#include "SyncState.h"
#include "Log.h"

#include "Config.h"

#include "BroadcastDef.h"

#include "WindowDef.h"

/*************
	prototypes
*************/
void Config(BOOL bStore);
/*************
	global vars
*************/
SyncState *g_pSyncState=NULL;
LogList *g_pLogList=NULL;

volatile LONG g_nReferenceCount=0;
CRITICAL_SECTION g_csBreakpointList;
BOOL g_bGuiEnabled=TRUE;

HINSTANCE g_hModule;
HWND g_hWndMainFrame=NULL;
//wintruder plug-in data
const WPI_Data gWPIData={
	sizeof(WPI_Data),
	PLUGIN_NAME,
	PLUGIN_DESCRIPTION,
	PLUGIN_COPYRIGHT,
	0,
	PLUGIN_FLAGS,
	PLUGIN_DEPENDICES};

/*************
	dllmain and exported functions
*************/
BOOL APIENTRY DllMain(HINSTANCE hModule,DWORD ul_reason_for_call,LPVOID lpReserved)
{
	if(ul_reason_for_call==DLL_PROCESS_ATTACH)
	{
		g_hModule=hModule;
	}
    return TRUE;
}

/*! @brief plugin load callback
	@details
	<ul>
		<li>if the plugin is loaded</li>
		<ul>
			<li>store handle of wintruder window</li>
			<li>initialize global critial section g_csBreakpointList</li>
			<li>get RootSymbol</li>
			<li>get SyncState from RootSymbol</li>
			<li>get ConfigFile and restore the state of g_bGuiEnabled</li>
		</ul>
	</ul>
**/
LIBFNC_WPI const WPI_Data* WPI_Load(DWORD version,HWND hWndMainFrame,BOOL bTrigger,void *pReserved)
{
	if(!bTrigger)
	{
		g_hWndMainFrame=hWndMainFrame;
	//get root symbol
		SymbolPtr<Symbol> pRootSymbol=GetRootSymbol();
		if(!pRootSymbol)
			return NULL;
	//get sync state
		g_pSyncState=(SyncState*)pRootSymbol->ChildByIdentifer(SYM_MAKEID(SYMID_SYNCSTATE,0));
		if(!g_pSyncState)
			return NULL;
	//get log list
		g_pLogList=(LogList*)pRootSymbol->ChildByIdentifer(SYM_MAKEID(SYMID_LOGLIST,0));
		if(!g_pLogList)
		{
			g_pSyncState->Release();
			g_pSyncState=NULL;
			return NULL;
		}
	//get config state
		Config(FALSE);
	//initialize section
		InitializeCriticalSection(&g_csBreakpointList);
	}
	return (WPI_Data*)&gWPIData;
}
/*! @brief plugin unload callback
	@details just check if PlugIn can be unloaded
	<ul>
		<li>delete global critical section g_csBreakpointList</li>
		<li>relase global pointer to SyncState</li>
		<li>relase global pointer to LogList</li>
	</ul>
**/
LIBFNC_WPI BOOL WPI_Unload(BOOL bForce)
{
	if(!g_nReferenceCount||bForce)
	{
		if(g_hWndMainFrame)
		{
			DeleteCriticalSection(&g_csBreakpointList);
			if(g_pSyncState)
			{
				g_pSyncState->Release();
				g_pSyncState=NULL;
			}
			if(g_pLogList)
			{
				g_pLogList->Release();
				g_pLogList=NULL;
			}
		//set config state
			Config(TRUE);
			g_hWndMainFrame=NULL;
		}
		return TRUE;
	}
	return FALSE;
}

/*!	@brief plugin command callback
	@details insert PlugIn commands to Wintruders main menu
	<ul>
		<li>on @ref WPICMD_MENU, just return the menu string</li>
		<li>on @ref WPICMD_MENUCMD, toggle enabled state</li>
		<li>on @ref WPICMD_MENUUPDATE, update menu item according to enabled state</li>
	</ul>
**/
LIBFNC_WPI int WPI_Command(UINT cmdId,UINT winId,HWND hWndFrame,UINT item,char data[2048])
{
	if(winId==WIN_MAINFRAME)
	{
		if(cmdId==WPICMD_MENU)
		{
			strcpy(data,"+0&Gui integration");
			return 1;
		}
		if(cmdId==WPICMD_MENUCMD)
		{
			ASSERT(item==0);
			g_bGuiEnabled=!g_bGuiEnabled;
		}
		if(cmdId==WPICMD_MENUUPDATE)
		{
			ASSERT(item==0);
			return (g_bGuiEnabled)?MFS_CHECKED:0;
		}
	}
	return 0;
}
/*!	@brief plugin broadcast callback
	@details setup BreakpointListX86Function upon @ref H_PROCESS_ADD message
	<ul>
		<li>check if the process added is an executing process. if so its @ref SYM_SUBID is @ref PROC_PROCESS</li>
		<li>test if @ref BP_X86 breakpoints are available</li>
		<li>test if @ref BP_X86INDIRECT breakpoints are available</li>
		<li>finally add a new BreakpointListX86Function symbol to the Process</li>
	</ul>
**/
LIBFNC_WPI LRESULT WPI_Broadcast(WPARAM message,LPARAM data)
{
	if(message==H_PROCESS_ADD)
	{
		Process *pProcess=(Process*)data;
		ASSERT(pProcess);
		if(	pProcess&&
			pProcess->Identifer()==SYM_MAKEID(SYMID_PROCESS,PROC_PROCESS))
		{
			SymbolPtr<Symbol> pSymbol=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86));
			if(pSymbol)
			{
				pSymbol=pProcess->ChildByIdentifer(SYM_MAKEID(SYMID_BREAKPOINTLIST,BP_X86INDIRECT));
				if(pSymbol)
				{
					pSymbol=new BreakpointListX86Function(pProcess);
					pProcess->AddChild(pSymbol);
				}
			}
		}
	}
	return 0;
}

/*!	@brief handle configuration
	@param bStore TRUE if configuration is to be stored
	@details
	<ul>
		<li>check g_hWndMainFrame and test if plug in initialized</li>
		<li>get RootSymbol</li>
		<li>get ConfigList</li>
		<li>get ConfigFile</li>
		<li>load/store configuration</li>
	</ul>
*/
void Config(BOOL bStore)
{
	if(g_hWndMainFrame)
	{
		SymbolPtr<Symbol> pRootSymbol=GetRootSymbol();
		if(pRootSymbol)
		{
		//get the configfile and load hook state
			SymbolPtr<Symbol> pSym=pRootSymbol->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIGLIST,0));
			if(pSym)
			{
				pSym=pSym->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIG,CONFIG_FILE));
				if(pSym)
				{
					SymbolPtr<ConfigNode> pConfigRoot=pSym->ChildByIdentifer(SYM_MAKEID(SYMID_CONFIG,CONFIG_ROOT));
					if(pConfigRoot)
					{
					//load/store config
						pConfigRoot->Config("PlugIn\\" PLUGIN_NAME,"EnableGui",g_bGuiEnabled,TRUE,bStore);
					}
				}
			}
		}
	}
}
