/***************************
	wintruder plugin sample MapExport
	copyright future vision - all rights reserved
***************************/
/*! @mainpage Wintruder PlugIn SDK Sample MapExport
	@anchor MAIN_MapExport
	@section sample_about About MapExport
	MapExport generates map files from symbol information loaded by Wintruder.
	@section sample_demonstrates This sample demonstrates
	<ul>
		<li>Very basic gui command extension</li>
		<li>Access to Wintruders RootSymbol and Symbol tree</li>
		<li>Usage of Wintruders debug information</li>
		<ul>
			<li>Enumerating WintruderSymbol information</li>
			<li>Access the symbols WintruderType</li>
			<li>Access the symbols _sym_storage</li>
		</ul>
	</ul>
	@section sample_where_start Where to start ?
	<ul>
		<li>@ref MapExport.cpp "MapExport documentation"</li>
		<li><a href="_map_export_8cpp-source.html">MapExport source</a></li>
	</ul>
	<ul>
		<li>@ref WintruderSdk.h "Sdk header file"</li>
		<li>@ref PAGE_Hirarchy "Wintruder symbol hirarchy"</li>
	</ul>
**/
#include "StdAfx.h"

#include "resource.h"

#include "Symbol.h"
#include "WindowDef.h"

#include "WintruderSymbolDef.h"

#include "SyncState.h"
#include "DirContainer.h"

#include "ModuleInfo.h"
#include "Section.h"
#include "WintruderDbgFile.h"
#include "WintruderType.h"


#include "MainDef.h"

/*************
	prototypes
*************/
INT_PTR CALLBACK SelectDbgFileDlg(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);

/*************
	global vars
*************/
HINSTANCE g_hModule;
HWND g_hWndMainFrame;
//wintruder plug-in data
const WPI_Data gWPIData={
	sizeof(WPI_Data),
	PLUGIN_NAME,
	PLUGIN_DESCRIPTION,
	PLUGIN_COPYRIGHT,
	0,
	PLUGIN_FLAGS,
	PLUGIN_DEPENDICES};

/*************
	simple string class
*************/
class String
{
public:
	String();
	~String();
	
	void empty();
	int length() const;

	LPCSTR operator=(LPCSTR p);
	LPCSTR operator+=(LPCSTR p);
	operator LPCSTR() const;
private:
	int m_nLength;
	LPSTR m_pData;
};

String::String()
{
	m_nLength=0;
	m_pData=(LPSTR)malloc(1);
	m_pData[0]=0;
}
String::~String()
{
	free(m_pData);
}
void String::empty()
{
	m_nLength=0;
}
int String::length() const
{
	return m_nLength;
}
LPCSTR String::operator=(LPCSTR p)
{
	empty();
	return operator+=(p);
}
LPCSTR String::operator+=(LPCSTR p)
{
	if(p&&*p)
	{
		int len=m_nLength+strlen(p);
		m_pData=(LPSTR)realloc(m_pData,len+1);
		
		strcpy(m_pData+m_nLength,p);
		m_nLength=len;
	}
	return m_pData;
}
String::operator LPCSTR()const
{
	return m_pData;
}


/*************
	dllmain and exported functions
*************/
BOOL APIENTRY DllMain(HINSTANCE hModule,DWORD ul_reason_for_call,LPVOID lpReserved)
{
	if(ul_reason_for_call==DLL_PROCESS_ATTACH)
	{
		g_hModule=hModule;
	}
    return TRUE;
}

LIBFNC_WPI const WPI_Data* WPI_Load(DWORD version,HWND hWndMainFrame,BOOL bTrigger,void *pReserved)
{
	if(!bTrigger)
	{
		g_hWndMainFrame=hWndMainFrame;
	}
	return (WPI_Data*)&gWPIData;
}

LIBFNC_WPI int WPI_Command(UINT cmdId,UINT winId,HWND hWndFrame,UINT item,char data[2048])
{
	if(winId==WIN_MAINFRAME)
	{
		if(cmdId==WPICMD_MENU)
		{
			strcpy(data,"0 &Export map file...");
			return 1;
		}
		if(cmdId==WPICMD_MENUCMD)
		{
			ASSERT(item==0);
			DialogBox(g_hModule,MAKEINTRESOURCE(IDD_DIALOG),g_hWndMainFrame,SelectDbgFileDlg);
		}
	}
	return 0;
}

/*************
	helper getting information of WintruderSymbol
*************/
/**	@brief recursive function for getting symbol name
	@param pSymbol pointer to the WintruderSymbol
	@param name reference of name returned
**/
void GetFullSymbolName(Symbol *pSymbol,String &name)
{
	SymbolPtr<Symbol> pParent=pSymbol->Parent();
	if(	pParent&&
		SYM_ID(pParent->Identifer())==SYMID_SYMBOL)
	{
		GetFullSymbolName(pParent,name);
		name+="::";
	}
	name+=pSymbol->Name();
}
/**	@brief get required data of WintruderSymbol
	@details
	<ul>
	<li>get _sym_storage_data of @em pSymbol</li>
	<li>if storage is static</li>
		<ul>
		<li>get rva of @em pSymbol</li>
		<li>if rva is valid</li>
			<ul>
			<li>get the name of @em pSymbol using @ref GetFullSymbolName</li>
			</ul>
		</ul>
	</ul>
	@param pSymbol pointer to the WintruderSymbol
	@param name reference of name returned
	@param section reference of section returned
	@param offset reference of offset returned
	@param rva reference of rva returned
	@returns true if successful
**/
bool GetStaticSymbolData(WintruderSymbol *pSymbol,String &name,DWORD &section,DWORD &offset,DWORD &rva)
{
	const _sym_storage_data *pStorage=pSymbol->Storage();
	if(	pStorage&&
		pStorage->store.type==_sym_storage::typeStatic)
	{
		section=pStorage->all._static.section;
		offset=pStorage->all._static.offset;

		rva=pSymbol->RVA();
		if(rva!=-1)
		{
			name.empty();
			GetFullSymbolName(pSymbol,name);
			return true;
		}
	}
	return false;
}
/*************
	export functions
*************/
//procedure
/**	@brief export procedure item
	@details
	<ul>
	<li>try to get static symbol data, such as name and storage</li>
	<li>enumerate children of @em pSymbol</li>
	<li>if child is argument</li>
		<ul>
		<li>get WintruderType of argument</li>
		<li>add the argument's type name</li>
		<li>add the argument's name</li>
		</ul>
	<li>write result to @em outFile</li>
	</ul>
	@param outFile map output file
	@param pSymbol WintruderSymbol of type @ref _sym_typeProc to be exported
**/
void ExportProc(FILE *outFile,WintruderSymbol *pSymbol)
{
	String name;
	DWORD section,offset;
	DWORD rva;
	if(GetStaticSymbolData(pSymbol,name,section,offset,rva))
	{
		name+="(";
	//unfortunately the current version of analyzer does not generate arguments
		SymbolPtr<WintruderType> pType;
		SymbolPtr<WintruderSymbol> pArg;
		bool bFirstArg=true;
		POSITION pos=pSymbol->FirstChildPosition();
		while(pos)
		{
			pArg=pSymbol->NextChild(pos);
			if(SYM_SUBID(pArg->Identifer())==_sym_typeArg)
			{
				if(!bFirstArg)
					name+=",";
				bFirstArg=false;
				pType=pArg->QueryType();
				if(pType)
					name+=pType->Name();
				else
					name+="<unknown>";
				name+=" ";
				name+=pArg->Name();
			}
		}
		name+=")";
		fprintf(outFile,"%4.4x:%8.8x %-30s %8.8x f\n",section,offset,(LPCSTR)name,rva);
	}
}
//other
/**	@brief export label or data item
	@details
	<ul>
	<li>try to get static symbol data, such as name and storage</li>
	<li>write result to @em outFile</li>
	</ul>
	@param outFile map output file
	@param pSymbol WintruderSymbol of type _sym_typeLabel or @ref _sym_typeData to be exported
**/
void ExportOther(FILE *outFile,WintruderSymbol *pSymbol)
{
	String name;
	DWORD section,offset;
	DWORD rva;
	if(GetStaticSymbolData(pSymbol,name,section,offset,rva))
	{
		fprintf(outFile,"%4.4x:%8.8x %-30s %8.8x\n",section,offset,(LPCSTR)name,rva);
	}
}
//WintruderSymbol
/**	@brief export Symbols children
	@details
	<ul>
	<li>enumerate all children of @em pSymbol</li>
		<ul>
		<li>if @em pChild is udt, recursively call @ref ExportChildren of @em pChild</li>
		<li>if @em pChild is proc, call @ref ExportProc</li>
		<li>if @em pChild is data or label, call @ref ExportOther</li>
		</ul>
	</ul>
	@param outFile map output file
	@param pSymbol WintruderSymbol or WintruderDbgFile whose children are to be exported
**/
void ExportChildren(FILE *outFile,Symbol *pSymbol)
{
	SymbolPtr<WintruderSymbol> pChild;
	POSITION pos=pSymbol->FirstChildPosition();
	while(pos)
	{
		pChild=pSymbol->NextChild(pos);
		switch(SYM_SUBID(pChild->Identifer()))
		{
		case _sym_typeUdt:
			ExportChildren(outFile,pChild);
			break;
		case _sym_typeProc:
			ExportProc(outFile,pChild);
			break;
		case _sym_typeLabel:
		case _sym_typeData:
			ExportOther(outFile,pChild);
			break;
		}
	}
}
//section
/**	@brief write section information
	@details
	<ul>
	<li>enumerate all sections of @em pSectionList</li>
	<li>if @em pSection is a vaild section (section of pe-file)</li>
		<ul>
		<li>write section data to @em outFile</li>
		</ul>
	</ul>
	@param outFile map output file
	@param pSectionList SectionList
**/
void ExportSections(FILE *outFile,Symbol *pSectionList)
{
	SymbolPtr<Section> pSection;
	POSITION pos=pSectionList->FirstChildPosition();
	while(pos)
	{
		pSection=pSectionList->NextChild(pos);
		if(!(pSection->Index()&(1<<31)))
		{
			fprintf(outFile,"%4.4x:%8.8x %8.8I64xH %-20s %s\n",
				pSection->Index(),0,
				pSection->Size(),
				pSection->Name(),
				(pSection->Spec()&IMAGE_SCN_CNT_CODE)?"CODE":"DATA");
		}
	}
}
//entrypoint
/**	@brief write entry point information
	@details
	<ul>
	<li>get the EntryPointList from @em pModuleInfo</li>
	<li>get module or dll EntryPoint form @em pEntryPointList</li>
		<ul>
		<li>get rva of @em pEntryPoint</li>
		<li>get section containing rva from @em pSectionList</li>
			<ul>
			<li>write section data to @em outFile</li>
			</ul>
		</ul>
	</ul>
	@param outFile map output file
	@param pModuleInfo ModuleInfo
	@param pSectionList SectionList
**/
void ExportEntryPoint(FILE *outFile,ModuleInfo *pModuleInfo,SectionList *pSectionList)
{
	SymbolPtr<Symbol> pEntryPointList=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_ENTRYPOINTLIST,0));
	if(pEntryPointList)
	{
		SymbolPtr<Symbol> pEntryPoint=pEntryPointList->ChildByIdentifer(SYM_MAKEID(SYMID_ENTRYPOINT,SYMDBG_MAKETYPE(DEBUG_X86,EP_MODULE)));
		if(!pEntryPoint)
			pEntryPoint=pEntryPointList->ChildByIdentifer(SYM_MAKEID(SYMID_ENTRYPOINT,SYMDBG_MAKETYPE(DEBUG_X86,EP_DLL)));
		if(pEntryPoint)
		{
			DWORD rva=pEntryPoint->RVA();
			if(rva!=-1)
			{
				SymbolPtr<Section> pSection=pSectionList->ChildByRVA(rva);
				if(pSection)
				{
					DWORD section=pSection->Index();
					DWORD offset=rva-pSection->RVA();
					fprintf(outFile,"entry point at %4.4x:%8.8x\n\n",section,offset);
				}
			}
		}
	}
}
//ModuleInfo (base of Section, WintruderSymbol)
/**	@brief export information of ModuleInfo
	@details
	<ul>
	<li>get SectionList from @em pModuleInfo</li>
		<ul>
		<li>get WintruderSymbolData from @em pSectionList</li>
			<ul>
			<li>get WintruderDbgFile of type @ref DEBUG_X86, @ref CREATOR_ANALYZER from @em pSymbolData</li>
				<ul>
				<li>generate the output file name and open file</li>
					<ul>
					<li>export basic information</li>
					<li>export section information using ExportSections</li>
					<li>export symbols contained in @em pDbgFile using ExportChildren</li>
					<li>export entry point information using ExportEntryPoint</li>
					</ul>
				</ul>
			</ul>
		</ul>
	</ul>
	@param pModuleInfo ModuleInfo
	@param exportPath
**/
void ExportModuleInfo(ModuleInfo *pModuleInfo,LPSTR exportPath)
{
	SymbolPtr<SectionList> pSectionList=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_SECTIONLIST,0));
	if(pSectionList)
	{
		SymbolPtr<Symbol> pSymbolData=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_SYMBOLDATA,0));
		if(pSymbolData)
		{
			SymbolPtr<WintruderDbgFile> pDbgFile=pSymbolData->ChildByIdentifer(SYM_MAKEID(SYMID_DBGFILE,SYMDBG_MAKETYPE(DEBUG_X86,CREATOR_ANALYZER)));
			if(pDbgFile)
			{
				char path[MAX_PATH];
				strcpy(path,exportPath);
				strcat(path,pModuleInfo->Name());
				strcat(path,".map");

				FILE *outFile=fopen(path,"wt");
				if(outFile)
				{
					fprintf(outFile,
						"%s\n\n"
						"Timestamp is %8.8x\n\n"
						"Preferred load address is %8.8I64x\n\n",
						pModuleInfo->Name(),
						pModuleInfo->TimeStamp(),
						pModuleInfo->DefaultImageBase());

					fprintf(outFile,"Start         Length    Name                 Class\n");
					ExportSections(outFile,pSectionList);
					fprintf(outFile,"\n");

					fprintf(outFile,"Address       Publics by Value               Rva+Base\n");
					ExportChildren(outFile,pDbgFile);
					fprintf(outFile,"\n");

					ExportEntryPoint(outFile,pModuleInfo,pSectionList);
					fclose(outFile);
				}
			}
		}
	}
}

/*************
	dialog functions
*************/
/**	@brief initialise the dialogs module list and export path
	@details
	<ul>
	<li>get Wintruder's RootSymbol</li>
		<ul>
		<li>get ModuleInfoList from @em pRoot</li>
			<ul>
			<li>enumerate all ModuleInfo contained in @em pModuleInfoList</li>
				<ul>
				<li>get SectionList from @em pModuleInfo</li>
					<ul>
					<li>get WintruderSymbolData from @em pModuleInfo</li>
						<ul>
						<li>get WintruderDbgFile of type DEBUG_X86, CREATOR_ANALYZER from @em pSymbolData</li>
							<ul>
							<li>add the path of @em pModuleInfo to list</li>
							</ul>
						</ul>
					</ul>
				</ul>
			</ul>
		<li>get ToolBox from @em pRoot</li>
			<ul>
			<li>get DirContainer from @em pToolBox</li>
				<ul>
				<li>get path stored in @em pDirContainer</li>
				</ul>
			</ul>
		<li>set default export path</li>
		</ul>
	</ul>
	@param hwndDlg handle of dialog box
**/
void OnInitDialog(HWND hwndDlg)
{
//init list
	HWND hWndList=GetDlgItem(hwndDlg,IDC_LIST);
	SymbolPtr<Symbol> pRoot=GetRootSymbol();
	if(pRoot)
	{
		SymbolPtr<Symbol> pModuleInfoList=pRoot->ChildByIdentifer(SYM_MAKEID(SYMID_MODULEINFOLIST,0));
		if(pModuleInfoList)
		{
			SymbolPtr<ModuleInfo> pModuleInfo;
			SymbolPtr<Symbol> pSectionList;
			SymbolPtr<Symbol> pSymbolData;
			SymbolPtr<WintruderDbgFile> pDbgFile;

			POSITION pos=pModuleInfoList->FirstChildPosition();
			while(pos)
			{
				pModuleInfo=pModuleInfoList->NextChild(pos);
				pSectionList=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_SECTIONLIST,0));
				if(pSectionList)
				{
					pSymbolData=pModuleInfo->ChildByIdentifer(SYM_MAKEID(SYMID_SYMBOLDATA,0));
					if(pSymbolData)
					{
						pDbgFile=pSymbolData->ChildByIdentifer(SYM_MAKEID(SYMID_DBGFILE,SYMDBG_MAKETYPE(DEBUG_X86,CREATOR_ANALYZER)));
						if(pDbgFile)
						{
							SendMessage(hWndList,LB_ADDSTRING,0,(LPARAM)pModuleInfo->Path());
						}
					}
				}
			}
		}
	}
//init path
	LPCSTR pPath=NULL;
	SymbolPtr<Symbol> pToolBox=pRoot->ChildByIdentifer(SYM_MAKEID(SYMID_TOOLBOX,0));
	if(pToolBox)
	{
		SymbolPtr<DirContainer> pDirContainer=pToolBox->ChildByIdentifer(SYM_MAKEID(SYMID_TOOL,TOOL_DIRCONTAINER));
		if(pDirContainer)
            pPath=pDirContainer->Get(PLUGIN_NAME);
	}
	if(!pPath)
		pPath="c:\\";
	HWND hWndPath=GetDlgItem(hwndDlg,IDC_PATH);
	SetWindowText(hWndPath,pPath);
}
/**	@brief export button handler
	@details
	<ul>
	<li>get Wintruder's RootSymbol</li>
		<ul>
		<li>get the export path and add a missing backslash</li>
		<li>get all selected items in list box</li>
			<ul>
			<li>get ModuleInfoList from @em pRoot</li>
				<ul>
				<li>get SyncState from @em pRoot and delay message</li>
				<li>get next selected text (path of module) from list box</li>
				<li>get ModuleInfo from @em pModuleInfoList using the path name</li>
					<ul>
					<li>export information by calling ExportModuleInfo</li>
					</ul>
				<li>delete the delayed message from SyncState</li>
				</ul>
			</ul>
		<li>get ToolBox from @em pRoot</li>
			<ul>
			<li>get DirContainer from @em pToolBox</li>
				<ul>
				<li>store export path into @em pDirContainer</li>
				</ul>
			</ul>
		</ul>
	</ul>
	@param hwndDlg handle of dialog box
**/
void OnExport(HWND hwndDlg)
{
	SymbolPtr<Symbol> pRoot=GetRootSymbol();
	if(pRoot)
	{
		char path[MAX_PATH];
		char exportPath[MAX_PATH];

		HWND hWndPath=GetDlgItem(hwndDlg,IDC_PATH);
		GetWindowText(hWndPath,exportPath,sizeof(exportPath));
		LPSTR p=exportPath+strlen(exportPath);
		if(p>exportPath&&p[-1]!='\\')
		{
			*p++='\\';
			*p=0;
		}

		HWND hWndList=GetDlgItem(hwndDlg,IDC_LIST);
		int nSel=SendMessage(hWndList,LB_GETSELCOUNT,0,0);
		if(nSel)
		{
			SymbolPtr<ModuleInfoList> pModuleInfoList=pRoot->ChildByIdentifer(SYM_MAKEID(SYMID_MODULEINFOLIST,0));
			if(pModuleInfoList)
			{
				SymbolPtr<SyncState> pSyncState=pRoot->ChildByIdentifer(SYM_MAKEID(SYMID_SYNCSTATE,0));

				POSITION posBusyAction=pSyncState->BusyAddAction("Export map files");

				SymbolPtr<ModuleInfo> pModuleInfo;

				int *pItems=new int[nSel];
				SendMessage(hWndList,LB_GETSELITEMS,nSel,(LPARAM)pItems);
				for(int i=0;i<nSel;i++)
				{
					SendMessage(hWndList,LB_GETTEXT,pItems[i],(LPARAM)path);
					pModuleInfo=pModuleInfoList->ChildByPath(path);
					if(pModuleInfo)
					{
						ExportModuleInfo(pModuleInfo,exportPath);
					}
				}
				delete []pItems;
				pSyncState->BusyRemoveAction(posBusyAction);
			}
		}//nSel
		SymbolPtr<Symbol> pToolBox=pRoot->ChildByIdentifer(SYM_MAKEID(SYMID_TOOLBOX,0));
		if(pToolBox)
		{
			SymbolPtr<DirContainer> pDirContainer=pToolBox->ChildByIdentifer(SYM_MAKEID(SYMID_TOOL,TOOL_DIRCONTAINER));
			if(pDirContainer)
				pDirContainer->Set(PLUGIN_NAME,exportPath);
		}
	}//pRoot
	EndDialog(hwndDlg,IDOK);
}

/*************
	dialog proc
*************/
/**	@brief dialog proc
**/
INT_PTR CALLBACK SelectDbgFileDlg(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{
	case WM_INITDIALOG:
		OnInitDialog(hwndDlg);
		return TRUE;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			OnExport(hwndDlg);
			return TRUE;
		case IDCANCEL:
			EndDialog(hwndDlg,IDCANCEL);
			return TRUE;
		}
	}
	return FALSE;
}
