/***************************
	wintruder sdk
	copyright future vision - all rights reserved
	
	this is preliminary information and may be subject to change
***************************/
#pragma once

#include "WintruderSdk.h"
#include "SymbolDef.h"

#ifndef _MFC_VER
	#define	CPropertyPage void
#endif


class Process;

/*!	@brief	top parent symbol class.

	%Symbol is the base class of all symbols and objects managed by wintruder.
	%Symbol provides a basic reference counting and auto deletion.
**/
class Symbol
{
public:
/*************
	introducing virtual
*************/
//destructor
	virtual ~Symbol();

//reference counting
	/*!	@brief increase the reference counter.
		@return resulting reference count.
	**/
	virtual ULONG __stdcall AddRef()=0;
	/*!	@brief decrease the reference counter.
		@return resulting reference count. if this value is 0, the symbol is destructed.
	**/
	virtual ULONG __stdcall Release()=0;
	/*!	@brief decrease the reference counter and invalidate reference chain.
		@return resulting reference count. if this value is 0, the symbol is destructed.
		@remarks - called by the parent during final Release() if the parent maintains a reference to the child
		@remarks - called by the child during final Release() if the child maintains a backreference to the parent
	**/
	virtual ULONG ReleaseEx(Symbol *pSym)=0;
//identifer
	/*!	@brief get the symbols identifer.
		@return symbols identifer. 
		the high dword is one of the @ref SYMID_xxx "SYMID_xxx" values and specifies the symbol type.
		the meaning of the low dword is symbol specific.
	**/
	virtual DWORDLONG Identifer()const=0;
//informational
	/*! @brief Get the symbol name.
		@return Returns a pointer to the name of the symbol. This value should never be NULL.
		@todo should be const
	**/
	virtual LPCSTR Name()=0;
	/*! @brief Get a desciptive text for the symbols type.
		@return Returns a pointer to the symbols type description. This value may be NULL.
		@todo should be const
	**/
	virtual LPCSTR Type()=0;
	/*! @brief Get a description for this symbol.
		@return Returns a pointer to the symbols description. This value may be NULL.
		@todo should be const
	**/
	virtual LPCSTR Description()=0;
//relatives
	/*! @brief Get the symbols parent.
		@return Returns a pointer to the symbols parent Symbol. This value may be NULL.
	**/
	virtual Symbol* Parent() const=0;

	/*! @brief Test if the symbol has children.
		@return Returns a none zero value, if the symbol has children.
	**/
	virtual BOOL HasChildren() const=0;
	/*! @brief Get the number of the symbols children.
		@return Returns the number of children.
	**/
	virtual int ChildrenCount() const=0;

	/*! @brief Add a child symbol.
		@param pSymbol A pointer to the %Symbol to be added 
		@return Returns a none zero value if successfully added.
	**/
	virtual BOOL AddChild(Symbol *pSymbol)=0;
	/*! @brief Remove a child symbol.
		@param pSymbol A pointer to the %Symbol to be removed.
		@return Returns a none zero value if successfully removed.
	**/
	virtual BOOL RemoveChild(Symbol *pSymbol)=0;

	/*! @brief Enumerate children.
		@return Returns a POSITION value used for enumerating the symbols children 
		or NULL if the symbol has no children.
	**/
	virtual POSITION FirstChildPosition()=0;
	/*! @brief Get the next child in the current enumeration.
		@param pos The enumerator returned by FirstChildPosition().
		@return Returns a pointer the next child. This value may be NULL.
		@remarks If the retrieved child is the last one, then the new value of 
		@em pos is set to NULL.
		@remarks The Identifer() member can be used to get the type of the %Symbol returned.
	**/
	virtual Symbol* NextChild(POSITION &pos)=0;
//size
	/*!	@brief Get the symbol size.
		@return Returns the symbols size or -1 of no size defined.
		@remarks Usually the size is the amount of memory occupied by this symbol.
	**/
	virtual DWORDLONG Size()=0;
//address
	/*!	@brief Get the symbols virtual start address.
		@param p_pt A pointer to a Process symbol. The virtual address in calculated
		in the context of this process.
		@return Returns the virtual address of the first memory location occupied by this symbol. 
		If there is no memory assigned or the virtual address cannot be
		calculated the function returns -1.
	**/
	virtual DWORDLONG VA(Process *p_pt)=0;
	/*!	@brief Get the symbols virtual end address.
		@param p_pt A pointer to a Process symbol. The virtual address in calculated
		in the context of this process.
		@return Returns the virtual address of the last memory location occupied by this symbol.
		If there is no memory assigned or the virtual address cannot be
		calculated the function returns -1.
		@remarks Some symbols have a start address only.
	**/
	virtual DWORDLONG VAEnd(Process *p_pt)=0;
	/*!	@brief Get the symbols virtual address range.
		@param start Retrieve the symbols virtual start address.
		@param end Retrieve the symbols virtual end address.
		@param p_pt A pointer to a Process symbol. The virtual addresses are calculated
		in the context of this process.
		@return Returns none zero if successfull.
		@remarks The @em start and @em end values are equal to the values returned by 
		VA() and VAEnd()
		@sa VA(), VAEnd()
	**/
	virtual BOOL VARange(DWORDLONG &start,DWORDLONG &end,Process *p_pt)=0;
	/*!	@brief Test if an address bounds within the virtual address range of this symbol.
		@param address The virtual address to be tested.
		@param p_pt A pointer to a process symbol. The virtual addresses of the symbol 
		are calculated in the context of this process.
		@return Returns none zero if @em address is within the virtual address range of the symbol.
		@remarks Some symbols return TRUE even if they have no @em start and/or @em end address assigned.
	**/
	virtual BOOL BoundsVA(DWORDLONG address,Process *p_pt)=0;

	/*!	@brief Get the symbols relative start address.
		@return Returns the relative address of the first memory location occupied by this symbol.
		If there is no memory assigned the function returns -1.
	**/
	virtual DWORD RVA()=0;
	/*!	@brief Get the symbols relative end address.
		@return Returns the relative address of the last memory location occupied by this symbol.
		If there is no memory assigned the function returns -1.
		@remarks Some symbols have a start address only.
	**/
	virtual DWORD RVAEnd()=0;
	/*!	@brief Get the symbols relative address range.
		@param start Retrieve the symbols relative start address.
		@param end Retrieve the symbols relative end address.
		@return Returns none zero if successfull.
		@remarks The @em start and @em end values are equal to the values returned by 
		RVA() and RVAEnd()
		@sa RVA(), RVAEnd()
	**/
	virtual BOOL RVARange(DWORD &start,DWORD &end)=0;
	/*!	@brief Test if an address bounds within the relative address range of this symbol.
		@param address The relative address to be tested.
		@return Returns none zero if @em address is within the relative address range of the symbol.
		@remarks Some symbols return TRUE even if they have no @em start and/or @em end address assigned.
	**/
	virtual BOOL BoundsRVA(DWORD address)=0;
//query children 
	/*!	@brief Get child symbol by identifer.
		@param id Identifer of the child to retrieve.
		@return Returns a pointer to a child symbol having the specified identifer.
		@remarks This function retrieves the first maching child only.
		@sa Identifer()
	**/
	virtual Symbol* ChildByIdentifer(DWORDLONG id)=0;
	/*!	@brief Get child symbol by name.
		@param name Pointer to the name of the child to retrieve.
		@return Returns a pointer to a child symbol having the specified name.
		@remarks This function retrieves the first maching child only.
		@sa Name()
	**/
	virtual Symbol* ChildByName(LPCSTR name)=0;

	/*!	@brief Get child symbol by virtual address.
		@param address Virtual address of the child to retrieve.
		@param p_pt A pointer to a process symbol. The virtual addresses of the symbol 
		are calculated in the context of this process.
		@return Returns a pointer to a child symbol containing the specified virtual address.
		@remarks This function retrieves the first maching child only.
		@sa VA()
	**/
	virtual Symbol* ChildByVA(DWORDLONG address,Process *p_pt)=0;
	/*!	@brief Get child symbol by relative address.
		@param address Relative address of the child to retrieve.
		@return Returns a pointer to a child symbol containing the specified relative address.
		@remarks This function retrieves the first maching child only.
		@sa RVA()
	**/
	virtual Symbol* ChildByRVA(DWORD address)=0;

	/*!	@brief Get child range by virtual address.
		@param address Virtual address of the child to retrieve.
		@param p_pt A pointer to a process symbol. The virtual addresses of the symbol 
		are calculated in the context of this process.
		@return Returns a pointer to a symbol containing the specified virtual address.
		@remarks If no child symbol contains @em address then a Range object is returned.
		The Range object spans the virtual address range of the gap containing @em address and
		not beeing occupied by any child symbol.
	**/
	virtual Symbol* ChildRangeByVA(DWORDLONG address,Process *p_pt)=0;
	/*!	@brief Get child range by relative address.
		@param address Relative address of the child to retrieve.
		@return Returns a pointer to a symbol containing the specified relative address.
		@remarks If no child symbol contains @em address then an Range object is returned.
		The Range object spans the relative address range of the gap containing @em address and
		not beeing occupied by any child symbol.
	**/
	virtual Symbol* ChildRangeByRVA(DWORD address)=0;
//symbol export
	/*!	@brief Export the symbol.
		@param p_pt Pointer to a Process symbol defining the context.
		@param pszFormat Pointer to a string containing the export format. The format string
		should be one of the strings returned by ExportList().
		@param pszFileName Pointer to a default file name. If @em pszFilename is NULL the symbol
		will browse for a file name.
		@return Returns none zero if successfull.
	**/
	virtual BOOL Export(Process *p_pt,LPCSTR pszFormat,LPCSTR pszFileName)=0;
	/*!	@brief Get available export formats.
		@param p_pt A pointer to the Process used as context.
		@param arFormat A pointer to a string pointer array retriving the available export formats.
		@param nFormat Number of free entries in @em arFormat
		@remarks The last (free) entry of arFormat will be set to NULL.
	**/
	virtual void ExportList(Process *p_pt,LPCSTR *arFormat,int nFormat)=0;

//todo
	virtual CPropertyPage* Property(DWORD dwThreadId)=0;

//sort/find
private:
	virtual void* Reserved_01(DWORD,const void*,void*,void*,void*,void*,int)
	{
		return NULL;
	}
	virtual void* Reserved_02(DWORD,const void*,void*,void*,void*,void*)
	{
		return NULL;
	}
	virtual void* Reserved_03(DWORD,BOOL&,void*)
	{
		return NULL;
	}
};


/*!	@brief	back reference Symbol
	@details this class is introduced for informational purposes only.
	any symbol derived from SymbolBackRef maintains a backward reference to its parent, 
	instead of using a foreward reference from parent to child.
	@n @n Foreward reference (note that the dashed backward reference does not maintain a reference counter)
@dot
digraph G {
	rankdir="RL"
	node [shape="box" style="filled" fillcolor="yellow" fontsize="8" height="0.3" width=0.5]
	Child->Parent [style=dashed color=blue]
	Parent->Child
}
@enddot
	@n Backward reference (note that the parent does not keep references to it's children)
@dot
digraph G {
	rankdir="RL"
	node [shape="box" style="filled" fillcolor="yellow" fontsize="8" height="0.3" width=0.5]
	Child->Parent
}
@enddot
	@warning if a symbol returns children that are derived from SymbolBackRef,
	different instances of the same object may be located at different locations in memory. 
	therefore the expression @code pSymbolBackRef_1 != pSymbolBackRef_2 @endcode has no meaning at all.
**/
class SymbolBackRef : public Symbol
{
public:
/*************
	introducing virtual
*************/
};

template <class T>
class SymbolPtr
{
public:
	inline SymbolPtr()
	{
		p=NULL;
	}
	inline SymbolPtr(Symbol *pSym)
	{
		p=NULL;
		operator=(pSym);
	}
	~SymbolPtr()
	{
		Release();
	}
#ifdef _DEBUG
	T* operator=(SymbolPtr<T> &o)
	{
		ASSERT(FALSE);
		return p;
	}
#endif
	inline T* operator=(Symbol *pSym)
	{
		Attach(pSym);
		return p;
	}
	inline bool operator!() const
	{
		return (p==NULL);
	}
	inline operator T*() const
	{
		return p;
	}
	inline T* operator->() const throw()
	{
		ASSERT(p!=NULL);
		return p;
	}
	inline void Release()
	{
		if(p)
		{
			T *pTemp=p;
			p=NULL;
			pTemp->Release();
		}
	}
	inline T* AddRef()
	{
		if(p)
			p->AddRef();
		return p;
	}
private:
	inline void Attach(Symbol *pSym)
	{
		Release();
		p=reinterpret_cast<T*>(pSym);
	}
	inline T* Detach()
	{
		T* pt=p;
		p=NULL;
		return pt;
	}

	T *p;
};